import requests
import json
from typing import Dict, Optional, Any
from web3 import Web3
from eth_account import Account

class ZeroXV2Integration:
    """
    Интеграция с 0x API v2 для обмена токенов
    Поддерживает AllowanceHolder flow согласно документации
    """
    
    def __init__(
        self,
        api_key: str = "",
        rpc_url: str = None,
        fee_recipient: Optional[str] = None,
        buy_token_percentage_fee: float = 0.0,
        sell_token_percentage_fee: float = 0.0,
        affiliate_address: Optional[str] = None
    ):
        self.api_key = api_key
        self.rpc_url = rpc_url
        self.base_url = "https://api.0x.org/swap/allowance-holder"  # 0x API v2 endpoint
        self.session = requests.Session()
        self.session.headers.update({
            '0x-api-key': self.api_key,
            '0x-version': 'v2',
            'Content-Type': 'application/json'
        })
        self.fee_recipient = fee_recipient
        self.buy_token_percentage_fee = buy_token_percentage_fee or 0.0
        self.sell_token_percentage_fee = sell_token_percentage_fee or 0.0
        self.affiliate_address = affiliate_address
    
    def update_rpc_url(self, rpc_url: str):
        """Обновляет RPC URL"""
        self.rpc_url = rpc_url
    
    def update_fee_config(self, *, fee_recipient=None, buy_fee=None, sell_fee=None, affiliate_address=None):
        """Динамически обновляет настройки комиссий"""
        if fee_recipient is not None:
            self.fee_recipient = fee_recipient
        if buy_fee is not None:
            self.buy_token_percentage_fee = buy_fee
        if sell_fee is not None:
            self.sell_token_percentage_fee = sell_fee
        if affiliate_address is not None:
            self.affiliate_address = affiliate_address
    
    def _apply_fee_params(self, params: Dict[str, Any]):
        """Добавляет параметры комиссии интегратора в запрос, если они настроены"""
        fee_applied = False
        if self.fee_recipient:
            if self.buy_token_percentage_fee and self.buy_token_percentage_fee > 0:
                params["buyTokenPercentageFee"] = f"{self.buy_token_percentage_fee:.8f}".rstrip('0').rstrip('.')
                fee_applied = True
            if self.sell_token_percentage_fee and self.sell_token_percentage_fee > 0:
                params["sellTokenPercentageFee"] = f"{self.sell_token_percentage_fee:.8f}".rstrip('0').rstrip('.')
                fee_applied = True
            if fee_applied:
                params["feeRecipient"] = self.fee_recipient
        if self.affiliate_address:
            params["affiliateAddress"] = self.affiliate_address
        return params
    
    def get_price(self, sell_token: str, buy_token: str, sell_amount: float, chain_id: int, taker: str = None, sell_decimals: int = 18) -> Optional[Dict]:
        """
        Получает индикативную цену от 0x API v2
        
        Args:
            sell_token: Адрес токена для продажи (или 0xeee... для нативного токена)
            buy_token: Адрес токена для покупки
            sell_amount: Количество токена для продажи (в токенах, не в wei)
            chain_id: ID сети
            taker: Адрес получателя (опционально)
            sell_decimals: Количество decimals у токена продажи (по умолчанию 18)
        """
        try:
            # Конвертируем amount в минимальные единицы с учетом decimals
            sell_amount_units = int(sell_amount * (10**sell_decimals))
            
            params = {
                "chainId": str(chain_id),
                "sellToken": sell_token,
                "buyToken": buy_token,
                "sellAmount": str(sell_amount_units)
            }
            
            # taker опционален для price endpoint (согласно документации)
            if taker:
                params["taker"] = taker
            
            params = self._apply_fee_params(params)
            
            url = f"{self.base_url}/price"
            print(f"🔍 Запрашиваем цену от 0x API v2: {url}")
            print(f"📋 Параметры: chainId={chain_id}, sellToken={sell_token}, buyToken={buy_token}, sellAmount={sell_amount_units} ({sell_amount} токенов)")
            
            response = self.session.get(url, params=params, timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                print(f"✅ 0x API v2 Price ответ получен")
                print(f"📊 buyAmount: {data.get('buyAmount')}, sellAmount: {data.get('sellAmount')}")
                return data
            else:
                print(f"❌ Ошибка 0x API v2 Price: {response.status_code} - {response.text[:300]}")
                return None
                
        except Exception as e:
            print(f"❌ Ошибка получения цены от 0x API v2: {e}")
            import traceback
            traceback.print_exc()
            return None
    
    def get_quote(self, sell_token: str, buy_token: str, sell_amount: float, chain_id: int, taker: str, sell_decimals: int = 18) -> Optional[Dict]:
        """
        Получает финальную котировку от 0x API v2
        
        Args:
            sell_token: Адрес токена для продажи (или 0xeee... для нативного токена)
            buy_token: Адрес токена для покупки
            sell_amount: Количество токена для продажи (в токенах, не в wei)
            chain_id: ID сети
            taker: Адрес получателя (обязательно)
            sell_decimals: Количество decimals у токена продажи (по умолчанию 18)
        """
        try:
            # Конвертируем amount в минимальные единицы с учетом decimals
            sell_amount_units = int(sell_amount * (10**sell_decimals))
            
            params = {
                "chainId": str(chain_id),
                "sellToken": sell_token,
                "buyToken": buy_token,
                "sellAmount": str(sell_amount_units),
                "taker": taker,
                "slippageBps": "100"  # 1% slippage (в базисных пунктах: 100 = 1%)
            }
            params = self._apply_fee_params(params)
            
            url = f"{self.base_url}/quote"
            print(f"🔍 Запрашиваем котировку от 0x API v2: {url}")
            print(f"📋 Параметры: chainId={chain_id}, sellToken={sell_token}, buyToken={buy_token}, sellAmount={sell_amount_units} ({sell_amount} токенов), taker={taker}")
            
            response = self.session.get(url, params=params, timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                print(f"✅ 0x API v2 Quote ответ получен")
                print(f"📊 buyAmount: {data.get('buyAmount')}, sellAmount: {data.get('sellAmount')}")
                print(f"📊 minBuyAmount: {data.get('minBuyAmount')}")
                
                # Выводим tokenMetadata если доступно
                if 'tokenMetadata' in data:
                    print(f"📋 tokenMetadata: {data.get('tokenMetadata')}")
                
                # Выводим fees если доступны
                if 'fees' in data:
                    print(f"💰 fees: {data.get('fees')}")
                
                # Выводим transaction info
                if 'transaction' in data:
                    tx = data.get('transaction')
                    print(f"⛽ gas: {tx.get('gas')}, gasPrice: {tx.get('gasPrice')}, value: {tx.get('value')}")
                
                return data
            else:
                print(f"❌ Ошибка 0x API v2 Quote: {response.status_code} - {response.text[:300]}")
                return None
                
        except Exception as e:
            print(f"❌ Ошибка получения котировки от 0x API v2: {e}")
            import traceback
            traceback.print_exc()
            return None
    
    def execute_swap(self, private_key: str, quote_data: Dict) -> Dict:
        """
        Выполняет обмен используя данные котировки от 0x API v2
        """
        try:
            if not self.rpc_url:
                return {"success": False, "error": "RPC URL не настроен"}
            
            # Создаем Web3 подключение
            web3 = Web3(Web3.HTTPProvider(self.rpc_url))
            if not web3.is_connected():
                return {"success": False, "error": "Не удалось подключиться к RPC"}
            
            # Создаем аккаунт из приватного ключа
            account = Account.from_key(private_key)
            
            # Получаем данные транзакции из котировки
            transaction_data = quote_data.get('transaction', {})
            if not transaction_data:
                return {"success": False, "error": "Данные транзакции не найдены в котировке"}
            
            # Подготавливаем транзакцию
            tx_params = {
                'from': account.address,
                'to': transaction_data.get('to'),
                'data': transaction_data.get('data'),
                'value': int(transaction_data.get('value', 0)),
                'gas': int(transaction_data.get('gas', 300000)),
                'gasPrice': int(transaction_data.get('gasPrice', web3.eth.gas_price)),
                'nonce': web3.eth.get_transaction_count(account.address)
            }
            
            print(f"🔄 Выполняем обмен для адреса: {account.address}")
            print(f"📋 Параметры транзакции: {tx_params}")
            
            # Подписываем транзакцию
            signed_txn = web3.eth.account.sign_transaction(tx_params, private_key)
            
            # Отправляем транзакцию
            tx_hash = web3.eth.send_raw_transaction(signed_txn.rawTransaction)
            tx_hash_hex = tx_hash.hex()
            
            print(f"✅ Транзакция отправлена: {tx_hash_hex}")
            
            return {
                "success": True,
                "tx_hash": tx_hash_hex,
                "transaction_url": f"https://etherscan.io/tx/{tx_hash_hex}"  # Будет обновлено для разных сетей
            }
            
        except Exception as e:
            print(f"❌ Ошибка выполнения обмена: {e}")
            return {"success": False, "error": str(e)}
    
    def get_supported_chains(self) -> Optional[Dict]:
        """
        Получает список поддерживаемых сетей от 0x API
        """
        try:
            url = "https://api.0x.org/swap/chains"
            headers = {
                '0x-api-key': self.api_key,
                '0x-version': 'v2'
            }
            
            response = requests.get(url, headers=headers, timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                print(f"✅ Получен список поддерживаемых сетей: {len(data)} сетей")
                return data
            else:
                print(f"❌ Ошибка получения списка сетей: {response.status_code}")
                return None
                
        except Exception as e:
            print(f"❌ Ошибка получения списка сетей: {e}")
            return None
    
    def check_allowance(self, token_address: str, owner: str, spender: str, chain_id: int) -> Optional[Dict]:
        """
        Проверяет allowance для токена
        """
        try:
            if not self.rpc_url:
                return None
                
            web3 = Web3(Web3.HTTPProvider(self.rpc_url))
            if not web3.is_connected():
                return None
            
            # ERC20 ABI для allowance
            erc20_abi = [
                {
                    "constant": True,
                    "inputs": [
                        {"name": "_owner", "type": "address"},
                        {"name": "_spender", "type": "address"}
                    ],
                    "name": "allowance",
                    "outputs": [{"name": "", "type": "uint256"}],
                    "type": "function"
                }
            ]
            
            contract = web3.eth.contract(
                address=Web3.to_checksum_address(token_address),
                abi=erc20_abi
            )
            
            allowance = contract.functions.allowance(
                Web3.to_checksum_address(owner),
                Web3.to_checksum_address(spender)
            ).call()
            
            return {
                "allowance": allowance,
                "allowance_formatted": allowance / (10**18)  # Предполагаем 18 decimals
            }
            
        except Exception as e:
            print(f"❌ Ошибка проверки allowance: {e}")
            return None
    
    def get_token_prices(self, tokens: list, chain_id: int = 137) -> Dict[str, float]:
        """Получает курсы токенов через 0x API"""
        prices = {}
        
        try:
            # Используем USDC как базовую валюту для получения курсов
            base_token = "0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174"  # USDC на Polygon
            
            for token_symbol in tokens:
                try:
                    # Получаем адрес токена
                    token_address = self._get_token_address(token_symbol, chain_id)
                    if not token_address:
                        continue
                    
                    # Получаем курс через price endpoint (используем правильный v2 endpoint)
                    url = f"{self.base_url}/price"
                    params = {
                        'chainId': str(chain_id),
                        'sellToken': token_address,
                        'buyToken': base_token,
                        'sellAmount': '1000000000000000000',  # 1 токен (18 decimals)
                    }
                    
                    response = self.session.get(url, params=params, timeout=10)
                    response.raise_for_status()
                    
                    data = response.json()
                    # В v2 API структура ответа: buyAmount и sellAmount, а не price
                    if 'buyAmount' in data and 'sellAmount' in data:
                        # Конвертируем: buyAmount (в минимальных единицах buyToken, т.е. USDC с 6 decimals)
                        # sellAmount (в минимальных единицах sellToken, т.е. токен с 18 decimals)
                        buy_amount = int(data['buyAmount'])  # Минимальные единицы USDC
                        sell_amount = int(data['sellAmount'])  # Минимальные единицы токена
                        # Цена = сколько USDC за 1 токен
                        # buyAmount уже в минимальных единицах USDC (6 decimals)
                        # sellAmount уже в минимальных единицах токена (18 decimals)
                        price_usd = (buy_amount / (10 ** 6)) / (sell_amount / (10 ** 18))
                        prices[token_symbol] = price_usd
                        print(f"✅ Курс {token_symbol} через 0x API v2: ${price_usd:.6f}")
                    elif 'price' in data:
                        # Fallback на старый формат (если есть)
                        price_usd = float(data['price']) / (10 ** 6)  # USDC имеет 6 decimals
                        prices[token_symbol] = price_usd
                        print(f"✅ Курс {token_symbol} через 0x API (legacy): ${price_usd:.6f}")
                    
                except Exception as e:
                    print(f"⚠️ Ошибка получения курса {token_symbol}: {e}")
                    continue
                    
        except Exception as e:
            print(f"❌ Ошибка получения курсов токенов: {e}")
        
        return prices
    
    def get_available_tokens(self, chain_id: int) -> list:
        """Получает список доступных токенов для указанной сети"""
        token_addresses = self._get_token_addresses_map()
        return list(token_addresses.get(chain_id, {}).keys())
    
    def _get_token_addresses_map(self) -> dict:
        """Возвращает полный маппинг токенов для всех сетей"""
        return {
            137: {  # Polygon
                "POL": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",
                "MATIC": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",
                "USDT": "0xc2132D05D31c914a87C6611C10748AEb04B58e8F",
                "USDC": "0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174",
                "WBTC": "0x1BFD67037B42Cf73acF2047067bd4F2C47D9BfD6",
                "DAI": "0x8f3Cf7ad23Cd3CaDbD9735AFf958023239c6A063",
                "WETH": "0x7ceB23fD6bC0adD59E62ac25578270cFf1b9f619",
                "ETH": "0x7ceB23fD6bC0adD59E62ac25578270cFf1b9f619",
                "LINK": "0x53E0bca35eC356BD5ddDFebbD1Fc0fD03FaBad39",
                "AAVE": "0xD6DF932A45C0f255f85145f286eA0b292B21C90B",
                "UNI": "0xb33EaAd8d922B1083446DC23f610c2567fB5180f",
                "CRV": "0x172370d5Cd63279eFa6d502DAB2917191a842FF4",
                "1INCH": "0x9c2C5fd7b07E95EE044DDeba0E97a665F142394f",
                "SUSHI": "0x0b3F868E0BE5597D5DB7fEB59E1CADBb0fdDa50a",
                "BAL": "0x9a71012B13CA4d3D0Cdc72A177DF3ef03b0E76a3",
                "YFI": "0xDA537104D6A5edd53c6FdBf9E3D95B6B2c6D8C79",
            },
            1: {  # Ethereum
                "ETH": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",
                "USDT": "0xdAC17F958D2ee523a2206206994597C13D831ec7",
                "USDC": "0xA0b86991c6218b36c1d19D4a2e9Eb0cE3606eB48",
                "WBTC": "0x2260FAC5E5542a773Aa44fBCfeDf7C193bc2C599",
                "DAI": "0x6B175474E89094C44Da98b954EedeAC495271d0F",
                "LINK": "0x514910771AF9Ca656af840dff83E8264EcF986CA",
            },
            56: {  # BSC
                "BNB": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",
                "USDT": "0x55d398326f99059fF775485246999027B3197955",
                "USDC": "0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d",
                "DAI": "0x1AF3F329e8BE154074D8769D1FFa4eE058B1DBc3",
                "WBTC": "0x7130d2A12B9BCbFAe4f2634d864A1Ee1Ce3Ead9c",
                "LINK": "0xF8A0BF9cF54Bb92F17374d9e9A321E6a111a51bD",
            },
            42161: {  # Arbitrum
                "ETH": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",
                "USDT": "0xFd086bC7CD5C481DCC9C85ebE478A1C0b69FCbb9",
                "USDC": "0xFF970A61A04b1cA14834A43f5dE4533eBDDB5CC8",
                "DAI": "0xDA10009cBd5D07dd0CeCc66161FC93D7c9000da1",
                "WBTC": "0x2f2a2543B76A4166549F7aaB2e75Bef0aefC5B0f",
                "LINK": "0xf97f4df75117a78c1A5a0DBb814Af92458539FB4",
            },
            10: {  # Optimism
                "ETH": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",
                "USDT": "0x94b008aA00579c1307B0EF2c499aD98a8ce58e58",
                "USDC": "0x7F5c764cBc14f9669B88837ca1490cCa17c31607",
                "DAI": "0xDA10009cBd5D07dd0CeCc66161FC93D7c9000da1",
                "WBTC": "0x68f180fcCe6836688e9084f035309E29Bf0A2095",
                "LINK": "0x350a791Bfc2C21F9Ed5d10980Dad2e2638ffa7f6",
            },
            43114: {  # Avalanche
                "AVAX": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",
                "USDT": "0x9702230A8Ea53601f5cD2dc00fDBc13d4dF4A8c7",
                "USDC": "0xB97EF9Ef8734C71904D8002F8b6Bc66Dd9c48a6E",
                "DAI": "0xd586E7F844cEa2F87f50152665BCbc2C279D8d70",
                "WBTC": "0x50b7545627a5162F82A992c33b87aDc75187B218",
                "LINK": "0x5947BB275c521040051D82396192181b413227A3",
            },
            8453: {  # Base
                "ETH": "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",
                "USDC": "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913",
                "DAI": "0x50c5725949A6F0c72E6C4a641F24049A917DB0Cb",
            }
        }
    
    def _get_token_address(self, symbol: str, chain_id: int) -> Optional[str]:
        """Получает адрес токена по символу"""
        token_addresses = self._get_token_addresses_map()
        return token_addresses.get(chain_id, {}).get(symbol.upper())
    
    def _get_token_decimals(self, symbol: str, chain_id: int, token_metadata: Dict = None) -> int:
        """
        Получает количество decimals для токена
        
        Args:
            symbol: Символ токена
            chain_id: ID сети
            token_metadata: Метаданные токена из ответа 0x API (если доступны)
                      Может быть dict (как в quote/price) или list
        """
        # Сначала пытаемся получить из tokenMetadata если доступно
        if token_metadata:
            # tokenMetadata может быть словарем с ключами 'buyToken' и 'sellToken'
            if isinstance(token_metadata, dict):
                # Ищем в обоих токенах
                for key in ['buyToken', 'sellToken']:
                    token_info = token_metadata.get(key, {})
                    if isinstance(token_info, dict):
                        token_symbol = token_info.get('symbol', '')
                        if token_symbol and token_symbol.upper() == symbol.upper():
                            decimals = token_info.get('decimals')
                            if decimals is not None:
                                return int(decimals)
            # Или списком токенов
            elif isinstance(token_metadata, list):
                for token_info in token_metadata:
                    if isinstance(token_info, dict) and token_info.get('symbol', '').upper() == symbol.upper():
                        decimals = token_info.get('decimals')
                        if decimals is not None:
                            return int(decimals)
        
        # Дефолтные decimals для известных токенов
        token_decimals = {
            137: {  # Polygon
                "POL": 18,
                "MATIC": 18,
                "USDT": 6,
                "USDC": 6,
                "WBTC": 8,
                "DAI": 18,
                "WETH": 18,
                "ETH": 18,
                "LINK": 18,
                "AAVE": 18,
                "UNI": 18,
                "CRV": 18,
                "1INCH": 18,
                "SUSHI": 18,
                "BAL": 18,
                "YFI": 18,
            },
            1: {  # Ethereum
                "ETH": 18,
                "USDT": 6,
                "USDC": 6,
                "WBTC": 8,
                "DAI": 18,
                "LINK": 18,
            },
            56: {  # BSC
                "BNB": 18,
                "USDT": 18,
                "USDC": 18,
                "DAI": 18,
                "WBTC": 18,
                "LINK": 18,
            },
            42161: {  # Arbitrum
                "ETH": 18,
                "USDT": 6,
                "USDC": 6,
                "DAI": 18,
                "WBTC": 8,
                "LINK": 18,
            },
            10: {  # Optimism
                "ETH": 18,
                "USDT": 6,
                "USDC": 6,
                "DAI": 18,
                "WBTC": 8,
                "LINK": 18,
            },
            43114: {  # Avalanche
                "AVAX": 18,
                "USDT": 6,
                "USDC": 6,
                "DAI": 18,
                "WBTC": 8,
                "LINK": 18,
            },
            8453: {  # Base
                "ETH": 18,
                "USDC": 6,
                "DAI": 18,
            }
        }
        
        return token_decimals.get(chain_id, {}).get(symbol.upper(), 18)  # По умолчанию 18
