from PyQt6.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QPushButton, QLabel, 
    QFrame, QLineEdit, QDialog, QMessageBox, QFileDialog,
    QSpacerItem, QSizePolicy, QGraphicsDropShadowEffect, QCheckBox, QTextEdit, QComboBox,
    QScrollArea
)
from PyQt6.QtCore import Qt, pyqtSignal, QTimer, QPropertyAnimation, QEasingCurve, QRect
from PyQt6.QtGui import QFont, QPixmap, QPainter, QColor, QPen, QBrush
import os
import json
import math
from typing import Optional

class WalletWelcomeUI(QWidget):
    """
    Начальное окно кошелька с опциями входа, создания и импорта.
    Включает голограмму в верхней части.
    """
    
    # Сигналы для взаимодействия с основным кошельком
    wallet_created = pyqtSignal(str, str)  # wallet_name, password
    wallet_imported = pyqtSignal(str, str, str)  # private_key, wallet_name, password
    wallet_unlocked = pyqtSignal(str, str)  # wallet_name, password
    
    def __init__(self, main_window=None):
        super().__init__()
        self.main_window = main_window
        self.setup_ui()
        self.setup_animations()
        
    def setup_ui(self):
        """Создание интерфейса приветственного окна"""
        self.setStyleSheet("""
            QWidget {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #1a1a2e, stop:1 #16213e);
                color: #ffffff;
                font-family: 'Segoe UI', Arial, sans-serif;
            }
            QFrame {
                background: rgba(255, 255, 255, 0.1);
                border: 1px solid rgba(255, 255, 255, 0.2);
                border-radius: 15px;
            }
            QPushButton {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #4CAF50, stop:1 #45a049);
                border: none;
                border-radius: 8px;
                color: white;
                font-weight: bold;
                font-size: 14px;
                padding: 12px 24px;
                min-height: 20px;
            }
            QPushButton:hover {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #5CBF60, stop:1 #55b059);
                transform: translateY(-2px);
            }
            QPushButton:pressed {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #3d8b40, stop:1 #357a38);
            }
            QPushButton#secondary_button {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #2196F3, stop:1 #1976D2);
            }
            QPushButton#secondary_button:hover {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #42A5F5, stop:1 #1E88E5);
            }
            QPushButton#danger_button {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #FF5722, stop:1 #E64A19);
            }
            QPushButton#danger_button:hover {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #FF7043, stop:1 #F4511E);
            }
            QLineEdit {
                background: rgba(255, 255, 255, 0.1);
                border: 2px solid rgba(255, 255, 255, 0.3);
                border-radius: 8px;
                padding: 12px;
                font-size: 14px;
                color: white;
            }
            QLineEdit:focus {
                border: 2px solid #4CAF50;
                background: rgba(255, 255, 255, 0.15);
            }
            QLabel {
                color: white;
                font-size: 14px;
            }
            QLabel#title_label {
                font-size: 28px;
                font-weight: bold;
                color: #4CAF50;
            }
            QLabel#subtitle_label {
                font-size: 16px;
                color: #a0a0a0;
            }
        """)
        
        # Основной layout
        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(20, 20, 20, 20)
        main_layout.setSpacing(20)
        
        # Создаем голограмму в верхней части
        self.create_hologram_section(main_layout)
        
        # Создаем основное содержимое
        self.create_main_content(main_layout)
        
    def create_hologram_section(self, parent_layout):
        """Создание секции с голограммой"""
        hologram_frame = QFrame()
        hologram_frame.setFixedHeight(120)
        hologram_frame.setStyleSheet("""
            QFrame {
                background: qlineargradient(x1:0, y1:0, x2:1, y2:1,
                    stop:0 rgba(76, 175, 80, 0.1), stop:1 rgba(33, 150, 243, 0.1));
                border: 2px solid rgba(76, 175, 80, 0.3);
                border-radius: 15px;
            }
        """)
        
        hologram_layout = QHBoxLayout(hologram_frame)
        hologram_layout.setContentsMargins(20, 10, 20, 10)
        
        # Левая часть - иконка/логотип
        icon_label = QLabel("🔐")
        icon_label.setStyleSheet("""
            QLabel {
                font-size: 48px;
                color: #4CAF50;
                background: transparent;
                border: none;
            }
        """)
        icon_label.setFixedSize(80, 80)
        
        # Центральная часть - текст
        text_layout = QVBoxLayout()
        text_layout.setSpacing(5)
        
        title_label = QLabel("AIagent Wallet")
        title_label.setObjectName("title_label")
        title_label.setStyleSheet("font-size: 24px; font-weight: bold; color: #4CAF50;")
        
        subtitle_label = QLabel("Безопасный Web3 кошелек")
        subtitle_label.setObjectName("subtitle_label")
        subtitle_label.setStyleSheet("font-size: 14px; color: #a0a0a0;")
        
        text_layout.addWidget(title_label)
        text_layout.addWidget(subtitle_label)
        text_layout.addStretch()
        
        # Правая часть - анимированная голограмма
        self.hologram_widget = HologramWidget()
        self.hologram_widget.setFixedSize(80, 80)
        
        hologram_layout.addWidget(icon_label)
        hologram_layout.addLayout(text_layout)
        hologram_layout.addWidget(self.hologram_widget)
        
        parent_layout.addWidget(hologram_frame)
        
    def create_main_content(self, parent_layout):
        """Создание основного содержимого с опциями"""
        self.main_content = QFrame()
        self.main_content.setStyleSheet("""
            QFrame {
                background: rgba(255, 255, 255, 0.05);
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 15px;
            }
        """)
        
        content_layout = QVBoxLayout(self.main_content)
        content_layout.setContentsMargins(30, 30, 30, 30)
        content_layout.setSpacing(20)
        
        # Заголовок
        welcome_label = QLabel("Добро пожаловать в AIagent Wallet")
        welcome_label.setStyleSheet("font-size: 20px; font-weight: bold; color: #ffffff;")
        welcome_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        
        desc_label = QLabel("Выберите действие для начала работы с кошельком")
        desc_label.setStyleSheet("font-size: 14px; color: #a0a0a0;")
        desc_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        
        content_layout.addWidget(welcome_label)
        content_layout.addWidget(desc_label)
        
        # Кнопки действий
        buttons_layout = QVBoxLayout()
        buttons_layout.setSpacing(15)
        
        # Кнопка "Создать новый кошелек"
        create_btn = QPushButton("🆕 Создать новый кошелек")
        create_btn.setObjectName("secondary_button")
        create_btn.clicked.connect(self.show_create_wallet_dialog)
        
        # Кнопка "Импортировать кошелек"
        import_btn = QPushButton("📥 Импортировать кошелек")
        import_btn.setObjectName("secondary_button")
        import_btn.clicked.connect(self.show_import_wallet_dialog)
        
        # Кнопка "Войти в существующий кошелек"
        unlock_btn = QPushButton("🔓 Войти в существующий кошелек")
        unlock_btn.clicked.connect(self.show_unlock_wallet_dialog)
        
        buttons_layout.addWidget(create_btn)
        buttons_layout.addWidget(import_btn)
        buttons_layout.addWidget(unlock_btn)
        
        content_layout.addLayout(buttons_layout)
        
        # Информационная секция
        info_frame = QFrame()
        info_frame.setStyleSheet("""
            QFrame {
                background: rgba(33, 150, 243, 0.1);
                border: 1px solid rgba(33, 150, 243, 0.3);
                border-radius: 10px;
            }
        """)
        
        info_layout = QVBoxLayout(info_frame)
        info_layout.setContentsMargins(15, 15, 15, 15)
        
        info_title = QLabel("ℹ️ Информация")
        info_title.setStyleSheet("font-size: 14px; font-weight: bold; color: #2196F3;")
        
        info_text = QLabel("""
• Создание нового кошелька - генерирует новый приватный ключ
• Импорт кошелька - восстановление из существующего приватного ключа
• Вход в кошелек - разблокировка сохраненного кошелька
        """)
        info_text.setStyleSheet("font-size: 12px; color: #a0a0a0; line-height: 1.4;")
        info_text.setWordWrap(True)
        
        info_layout.addWidget(info_title)
        info_layout.addWidget(info_text)
        
        content_layout.addWidget(info_frame)
        
        parent_layout.addWidget(self.main_content)
        
    def setup_animations(self):
        """Настройка анимаций"""
        self.animation_timer = QTimer()
        self.animation_timer.timeout.connect(self.update_hologram_animation)
        self.animation_timer.start(50)  # 20 FPS
        
    def update_hologram_animation(self):
        """Обновление анимации голограммы"""
        if hasattr(self, 'hologram_widget'):
            self.hologram_widget.update_animation()
            
    def show_create_wallet_dialog(self):
        """Показать модальное окно создания кошелька"""
        self.show_modal("create")
            
    def show_import_wallet_dialog(self):
        """Показать модальное окно импорта кошелька"""
        self.show_modal("import")
            
    def show_unlock_wallet_dialog(self):
        """Показать модальное окно входа в кошелек"""
        self.show_modal("unlock")
        
    def show_modal(self, modal_type):
        """Показать модальное окно в интерфейсе"""
        # Скрываем основное содержимое
        self.main_content.hide()
        
        # Создаем модальное окно
        if modal_type == "create":
            self.modal_widget = CreateWalletModal(self)
        elif modal_type == "import":
            self.modal_widget = ImportWalletModal(self)
        elif modal_type == "unlock":
            self.modal_widget = UnlockWalletModal(self)
        
        # Добавляем в layout
        self.layout().addWidget(self.modal_widget)
        
        # Подключаем сигналы
        self.modal_widget.back_clicked.connect(self.hide_modal)
        if modal_type == "create":
            self.modal_widget.wallet_created.connect(self.handle_wallet_created)
        elif modal_type == "import":
            self.modal_widget.wallet_imported.connect(self.handle_wallet_imported)
        elif modal_type == "unlock":
            self.modal_widget.wallet_unlocked.connect(self.handle_wallet_unlocked)
    
    def hide_modal(self):
        """Скрыть модальное окно"""
        if hasattr(self, 'modal_widget'):
            self.modal_widget.deleteLater()
            delattr(self, 'modal_widget')
        
        # Показываем основное содержимое
        self.main_content.show()
    
    def handle_wallet_created(self, wallet_name, password):
        """Обработка создания кошелька"""
        self.wallet_created.emit(wallet_name, password)
        self.hide_modal()
        
    def handle_wallet_imported(self, private_key, wallet_name, password):
        """Обработка импорта кошелька"""
        self.wallet_imported.emit(private_key, wallet_name, password)
        self.hide_modal()
        
    def handle_wallet_unlocked(self, wallet_name, password):
        """Обработка входа в кошелек"""
        self.wallet_unlocked.emit(wallet_name, password)
        self.hide_modal()


class HologramWidget(QWidget):
    """Виджет голограммы для отображения анимации"""
    
    def __init__(self):
        super().__init__()
        self.animation_phase = 0
        self.setStyleSheet("background: transparent; border: none;")
        
    def update_animation(self):
        """Обновление анимации"""
        self.animation_phase += 0.1
        if self.animation_phase > 2 * 3.14159:
            self.animation_phase = 0
        self.update()
        
    def paintEvent(self, event):
        """Отрисовка голограммы"""
        painter = QPainter(self)
        painter.setRenderHint(QPainter.RenderHint.Antialiasing)
        
        center_x = self.width() // 2
        center_y = self.height() // 2
        radius = min(self.width(), self.height()) // 3
        
        # Создаем градиентный круг
        for i in range(5):
            alpha = int(255 * (1 - i * 0.2))
            color = QColor(76, 175, 80, alpha)
            pen = QPen(color, 2)
            painter.setPen(pen)
            
            current_radius = radius + i * 5
            painter.drawEllipse(center_x - current_radius, center_y - current_radius,
                              current_radius * 2, current_radius * 2)
        
        # Анимированные частицы
        for i in range(8):
            angle = (self.animation_phase + i * 0.785) % (2 * 3.14159)
            x = center_x + int(radius * 0.7 * math.cos(angle))
            y = center_y + int(radius * 0.7 * math.sin(angle))
            
            color = QColor(33, 150, 243, 200)
            painter.setPen(QPen(color, 3))
            painter.drawPoint(x, y)


class CreateWalletModal(QWidget):
    """Модальное окно создания нового кошелька"""
    
    wallet_created = pyqtSignal(str, str)
    back_clicked = pyqtSignal()
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setup_ui()
        
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setSpacing(20)  # Уменьшаем spacing между элементами
        
        # Заголовок с кнопкой назад
        header_layout = QHBoxLayout()
        
        back_btn = QPushButton("← Назад")
        back_btn.setStyleSheet("""
            QPushButton {
                background: rgba(255, 255, 255, 0.1);
                border: 1px solid rgba(255, 255, 255, 0.3);
                border-radius: 8px;
                color: white;
                padding: 8px 16px;
                font-size: 14px;
            }
            QPushButton:hover {
                background: rgba(255, 255, 255, 0.2);
            }
        """)
        back_btn.clicked.connect(self.back_clicked.emit)
        
        title = QLabel("🆕 Создание нового кошелька")
        title.setStyleSheet("font-size: 20px; font-weight: bold; color: #4CAF50;")
        
        header_layout.addWidget(back_btn)
        header_layout.addWidget(title)
        header_layout.addStretch()
        
        # Форма
        form_frame = QFrame()
        form_frame.setStyleSheet("""
            QFrame {
                background: rgba(255, 255, 255, 0.05);
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 15px;
            }
        """)
        
        form_layout = QVBoxLayout(form_frame)
        form_layout.setContentsMargins(30, 30, 30, 30)
        form_layout.setSpacing(15)  # Уменьшаем spacing между полями ввода
        
        # Поля ввода
        self.name_input = QLineEdit()
        self.name_input.setPlaceholderText("Введите название кошелька")
        self.name_input.setFixedHeight(45)  # Увеличиваем высоту для текста
        
        self.password_input = QLineEdit()
        self.password_input.setEchoMode(QLineEdit.EchoMode.Password)
        self.password_input.setPlaceholderText("Введите пароль")
        self.password_input.setFixedHeight(45)  # Увеличиваем высоту для текста
        
        self.confirm_password_input = QLineEdit()
        self.confirm_password_input.setEchoMode(QLineEdit.EchoMode.Password)
        self.confirm_password_input.setPlaceholderText("Подтвердите пароль")
        self.confirm_password_input.setFixedHeight(45)  # Увеличиваем высоту для текста
        
        # Поле для отображения seed фразы - делаем выделяемым и копируемым
        self.seed_phrase_label = QLabel("Seed фраза будет показана после создания кошелька")
        self.seed_phrase_label.setStyleSheet("""
            QLabel {
                background: rgba(255, 255, 255, 0.1);
                border: 1px solid rgba(255, 255, 255, 0.2);
                border-radius: 8px;
                padding: 15px;
                color: #ffffff;
                font-size: 14px;
                min-height: 120px;
                max-height: 120px;
            }
        """)
        self.seed_phrase_label.setWordWrap(True)
        self.seed_phrase_label.setFixedHeight(120)  # Фиксированная высота для 12 слов
        self.seed_phrase_label.setTextInteractionFlags(Qt.TextInteractionFlag.TextSelectableByMouse | Qt.TextInteractionFlag.TextSelectableByKeyboard)
        self.seed_phrase_label.hide()  # Скрываем до создания кошелька
        
        # Чекбокс подтверждения записи seed фразы - убираем для устранения наложения
        # self.seed_confirmed_checkbox = QCheckBox("✅ Я записал seed фразу в безопасном месте")
        # self.seed_confirmed_checkbox.hide()  # Скрываем до показа seed фразы
        
        # Кнопка создания
        create_btn = QPushButton("🚀 Создать кошелек")
        create_btn.setFixedHeight(35)  # Уменьшаем высоту кнопки
        create_btn.setStyleSheet("""
            QPushButton {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #4CAF50, stop:1 #45a049);
                border: none;
                border-radius: 10px;
                color: white;
                font-weight: bold;
                font-size: 16px;
                padding: 15px;
                min-height: 20px;
            }
            QPushButton:hover {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #5CBF60, stop:1 #55b059);
            }
        """)
        create_btn.clicked.connect(self.create_wallet)
        
        # Добавляем элементы
        form_layout.addWidget(self.name_input)
        form_layout.addWidget(self.password_input)
        form_layout.addWidget(self.confirm_password_input)
        form_layout.addWidget(self.seed_phrase_label)
        # form_layout.addWidget(self.seed_confirmed_checkbox)  # Убираем чекбокс
        
        # Добавляем дополнительный отступ перед кнопкой
        spacer = QSpacerItem(20, 20, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Fixed)
        form_layout.addItem(spacer)
        
        form_layout.addWidget(create_btn)
        
        layout.addLayout(header_layout)
        layout.addWidget(form_frame)
        layout.addStretch()
        
    def create_wallet(self):
        """Создать кошелек"""
        name = self.name_input.text().strip()
        password = self.password_input.text()
        confirm_password = self.confirm_password_input.text()
        
        if not name:
            self.show_error("Введите название кошелька")
            return
            
        if not password:
            self.show_error("Введите пароль")
            return
            
        if password != confirm_password:
            self.show_error("Пароли не совпадают")
            return
            
        if len(password) < 6:
            self.show_error("Пароль должен содержать минимум 6 символов")
            return
        
        # Генерируем временный кошелек для показа seed фразы
        try:
            from plugins.web3_wallet import Web3Wallet
            temp_wallet = Web3Wallet()
            temp_wallet.account = temp_wallet.account or temp_wallet._create_temp_account()
            temp_wallet.seed_phrase = temp_wallet._generate_seed_phrase()
            
            # Показываем seed фразу
            seed_words = temp_wallet.seed_phrase.split()
            formatted_seed = " ".join(seed_words)
            self.seed_phrase_label.setText(f"🔐 Seed фраза ({len(seed_words)} слов):\n\n{formatted_seed}\n\n⚠️ Сохраните эту фразу в безопасном месте!")
            self.seed_phrase_label.show()
            # self.seed_confirmed_checkbox.show()  # Убираем показ чекбокса
            
            # Изменяем кнопку на "Подтвердить создание"
            sender = self.sender()
            sender.setText("✅ Подтвердить создание")
            sender.clicked.disconnect()
            sender.clicked.connect(lambda: self.confirm_wallet_creation(name, password, temp_wallet.seed_phrase))
            
        except Exception as e:
            self.show_error(f"Ошибка генерации seed фразы: {e}")
    
    def confirm_wallet_creation(self, name, password, seed_phrase):
        """Подтвердить создание кошелька"""
        # Убираем проверку чекбокса - сразу создаем кошелек
        # if not self.seed_confirmed_checkbox.isChecked():
        #     self.show_error("Пожалуйста, подтвердите, что вы записали seed фразу!")
        #     return
        self.wallet_created.emit(name, password)
    
    def show_error(self, message):
        """Показать ошибку"""
        # Простое уведомление об ошибке
        print(f"❌ Ошибка: {message}")


class ImportWalletModal(QWidget):
    """Модальное окно импорта кошелька"""
    
    wallet_imported = pyqtSignal(str, str, str)
    back_clicked = pyqtSignal()
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setup_ui()
        
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setSpacing(20)
        
        # Заголовок с кнопкой назад
        header_layout = QHBoxLayout()
        
        back_btn = QPushButton("← Назад")
        back_btn.setStyleSheet("""
            QPushButton {
                background: rgba(255, 255, 255, 0.1);
                border: 1px solid rgba(255, 255, 255, 0.3);
                border-radius: 8px;
                color: white;
                padding: 8px 16px;
                font-size: 14px;
            }
            QPushButton:hover {
                background: rgba(255, 255, 255, 0.2);
            }
        """)
        back_btn.clicked.connect(self.back_clicked.emit)
        
        title = QLabel("📥 Импорт кошелька")
        title.setStyleSheet("font-size: 20px; font-weight: bold; color: #2196F3;")
        
        header_layout.addWidget(back_btn)
        header_layout.addWidget(title)
        header_layout.addStretch()
        
        # Форма
        form_frame = QFrame()
        form_frame.setStyleSheet("""
            QFrame {
                background: rgba(255, 255, 255, 0.05);
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 15px;
            }
        """)
        
        form_layout = QVBoxLayout(form_frame)
        form_layout.setContentsMargins(30, 30, 30, 30)
        form_layout.setSpacing(10)  # Уменьшаем общий spacing
        
        # Поля ввода - убираем лейблы, используем только placeholder
        self.seed_phrase_input = QTextEdit()  # Изменяем на QTextEdit для многострочного ввода
        self.seed_phrase_input.setPlaceholderText("Введите seed фразу (12-24 слова)")
        self.seed_phrase_input.setFixedHeight(120)  # Увеличиваем высоту для многострочного отображения
        self.seed_phrase_input.setStyleSheet("""
            QTextEdit {
                background: rgba(255, 255, 255, 0.1);
                border: 1px solid rgba(255, 255, 255, 0.2);
                border-radius: 8px;
                color: #ffffff;
                font-size: 14px;
                padding: 15px;
                font-family: monospace;
            }
            QTextEdit:focus {
                border: 2px solid #2196F3;
                background: rgba(255, 255, 255, 0.15);
            }
        """)
        
        self.private_key_input = QLineEdit()
        self.private_key_input.setPlaceholderText("Введите приватный ключ")
        self.private_key_input.setFixedHeight(45)
        
        self.name_input = QLineEdit()
        self.name_input.setPlaceholderText("Введите название кошелька")
        self.name_input.setFixedHeight(45)
        
        self.password_input = QLineEdit()
        self.password_input.setEchoMode(QLineEdit.EchoMode.Password)
        self.password_input.setPlaceholderText("Введите пароль")
        self.password_input.setFixedHeight(45)
        
        # Кнопка импорта
        import_btn = QPushButton("📥 Импортировать кошелек")
        import_btn.setStyleSheet("""
            QPushButton {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #2196F3, stop:1 #1976D2);
                border: none;
                border-radius: 10px;
                color: white;
                font-weight: bold;
                font-size: 16px;
                padding: 15px;
                min-height: 20px;
            }
            QPushButton:hover {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #42A5F5, stop:1 #1E88E5);
            }
        """)
        import_btn.clicked.connect(self.import_wallet)
        
        # Добавляем элементы - убираем лейблы, добавляем seed фразу первым
        form_layout.addWidget(self.seed_phrase_input)
        
        # Добавляем БОЛЬШОЙ отступ после поля seed
        spacer_seed = QSpacerItem(20, 40, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Fixed)
        form_layout.addItem(spacer_seed)
        
        form_layout.addWidget(self.private_key_input)
        
        # Добавляем небольшой отступ между полями
        spacer1 = QSpacerItem(20, 10, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Fixed)
        form_layout.addItem(spacer1)
        
        form_layout.addWidget(self.name_input)
        
        # Добавляем небольшой отступ между полями
        spacer2 = QSpacerItem(20, 10, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Fixed)
        form_layout.addItem(spacer2)
        
        form_layout.addWidget(self.password_input)
        
        # Добавляем небольшой отступ между полями
        spacer3 = QSpacerItem(20, 10, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Fixed)
        form_layout.addItem(spacer3)
        
        form_layout.addWidget(import_btn)
        
        layout.addLayout(header_layout)
        layout.addWidget(form_frame)
        layout.addStretch()
        
    def import_wallet(self):
        """Импортировать кошелек"""
        seed_phrase = self.seed_phrase_input.toPlainText().strip()  # Используем toPlainText() для QTextEdit
        private_key = self.private_key_input.text().strip()
        name = self.name_input.text().strip()
        password = self.password_input.text()
        
        # Проверяем что введена либо seed фраза, либо приватный ключ
        if not seed_phrase and not private_key:
            self.show_error("Введите seed фразу или приватный ключ")
            return
            
        if not name:
            self.show_error("Введите название кошелька")
            return
            
        if not password:
            self.show_error("Введите пароль")
            return
        
        # Передаем seed фразу или приватный ключ
        # Определяем тип ввода по количеству слов
        if seed_phrase and len(seed_phrase.split()) > 1:
            # Это seed фраза (несколько слов)
            print(f"🔤 Определена seed фраза: {len(seed_phrase.split())} слов")
            self.wallet_imported.emit(seed_phrase, name, password)
        elif private_key:
            # Это приватный ключ
            print(f"🔑 Определен приватный ключ")
            self.wallet_imported.emit(private_key, name, password)
        else:
            # Если seed фраза содержит только одно слово, считаем это приватным ключом
            print(f"🔑 Seed фраза содержит одно слово, используем как приватный ключ")
            self.wallet_imported.emit(seed_phrase, name, password)
    
    def show_error(self, message):
        """Показать ошибку"""
        print(f"❌ Ошибка: {message}")


class UnlockWalletModal(QWidget):
    """Модальное окно входа в существующий кошелек"""
    
    wallet_unlocked = pyqtSignal(str, str)
    back_clicked = pyqtSignal()
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setup_ui()
        
    def setup_ui(self):
        layout = QVBoxLayout(self)
        layout.setSpacing(20)
        
        # Заголовок с кнопкой назад
        header_layout = QHBoxLayout()
        
        back_btn = QPushButton("← Назад")
        back_btn.setStyleSheet("""
            QPushButton {
                background: rgba(255, 255, 255, 0.1);
                border: 1px solid rgba(255, 255, 255, 0.3);
                border-radius: 8px;
                color: white;
                padding: 8px 16px;
                font-size: 14px;
            }
            QPushButton:hover {
                background: rgba(255, 255, 255, 0.2);
            }
        """)
        back_btn.clicked.connect(self.back_clicked.emit)
        
        title = QLabel("🔓 Вход в кошелек")
        title.setStyleSheet("font-size: 20px; font-weight: bold; color: #FF5722;")
        
        header_layout.addWidget(back_btn)
        header_layout.addWidget(title)
        header_layout.addStretch()
        
        # Форма
        form_frame = QFrame()
        form_frame.setStyleSheet("""
            QFrame {
                background: rgba(255, 255, 255, 0.05);
                border: 1px solid rgba(255, 255, 255, 0.1);
                border-radius: 15px;
            }
        """)
        
        form_layout = QVBoxLayout(form_frame)
        form_layout.setContentsMargins(30, 30, 30, 30)
        form_layout.setSpacing(20)
        
        # Выпадающий список кошельков с кнопкой удаления
        wallet_combo_layout = QHBoxLayout()
        wallet_combo_layout.setSpacing(10)
        
        self.wallet_combo = QComboBox()
        self.wallet_combo.setPlaceholderText("Выберите кошелек")
        self.wallet_combo.setStyleSheet("""
            QComboBox {
                background: rgba(255, 255, 255, 0.1);
                border: 1px solid rgba(255, 255, 255, 0.2);
                border-radius: 8px;
                color: #ffffff;
                font-size: 14px;
                padding: 10px;
                min-height: 20px;
            }
            QComboBox:hover {
                border: 2px solid #FF5722;
                background: rgba(255, 255, 255, 0.15);
            }
            QComboBox::drop-down {
                border: none;
                width: 30px;
            }
            QComboBox::down-arrow {
                image: none;
                border-left: 5px solid transparent;
                border-right: 5px solid transparent;
                border-top: 5px solid #ffffff;
                margin-right: 10px;
            }
            QComboBox QAbstractItemView {
                background: rgba(30, 30, 30, 0.95);
                border: 1px solid rgba(255, 255, 255, 0.2);
                border-radius: 8px;
                color: #ffffff;
                selection-background-color: #FF5722;
            }
        """)
        
        # Кнопка удаления кошелька
        self.delete_wallet_btn = QPushButton("🗑️")
        self.delete_wallet_btn.setToolTip("Удалить выбранный кошелек")
        self.delete_wallet_btn.setStyleSheet("""
            QPushButton {
                background: rgba(244, 67, 54, 0.7);
                border: 1px solid rgba(244, 67, 54, 0.9);
                border-radius: 8px;
                color: white;
                font-size: 16px;
                padding: 10px;
                min-width: 45px;
                max-width: 45px;
            }
            QPushButton:hover {
                background: rgba(244, 67, 54, 0.9);
            }
            QPushButton:disabled {
                background: rgba(100, 100, 100, 0.3);
                border: 1px solid rgba(100, 100, 100, 0.5);
            }
        """)
        self.delete_wallet_btn.clicked.connect(self._delete_selected_wallet)
        self.delete_wallet_btn.setEnabled(False)  # По умолчанию отключена
        
        # Обновляем состояние кнопки при изменении выбора
        self.wallet_combo.currentIndexChanged.connect(self._on_wallet_selected)
        
        wallet_combo_layout.addWidget(self.wallet_combo, 1)
        wallet_combo_layout.addWidget(self.delete_wallet_btn)
        
        # Загружаем список кошельков
        self.load_wallets()
        
        # Поле пароля (без надписи)
        # password_label = QLabel("Пароль:")
        # password_label.setStyleSheet("font-size: 14px; color: #ffffff; font-weight: bold;")
        self.password_input = QLineEdit()
        self.password_input.setEchoMode(QLineEdit.EchoMode.Password)
        self.password_input.setPlaceholderText("Введите пароль")
        self.password_input.setStyleSheet("""
            QLineEdit {
                background: rgba(255, 255, 255, 0.1);
                border: 1px solid rgba(255, 255, 255, 0.2);
                border-radius: 8px;
                color: #ffffff;
                font-size: 14px;
                padding: 10px;
                min-height: 20px;
            }
            QLineEdit:focus {
                border: 2px solid #FF5722;
                background: rgba(255, 255, 255, 0.15);
            }
        """)
        
        # Кнопка входа
        unlock_btn = QPushButton("🔓 Войти в кошелек")
        unlock_btn.setStyleSheet("""
            QPushButton {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #FF5722, stop:1 #E64A19);
                border: none;
                border-radius: 10px;
                color: white;
                font-weight: bold;
                font-size: 16px;
                padding: 15px;
                min-height: 20px;
            }
            QPushButton:hover {
                background: qlineargradient(x1:0, y1:0, x2:0, y2:1,
                    stop:0 #FF7043, stop:1 #F4511E);
            }
        """)
        unlock_btn.clicked.connect(self.unlock_wallet)
        
        # Добавляем элементы
        form_layout.addLayout(wallet_combo_layout)
        # form_layout.addWidget(password_label)  # Убрано
        form_layout.addWidget(self.password_input)
        form_layout.addWidget(unlock_btn)
        
        layout.addLayout(header_layout)
        layout.addWidget(form_frame)
        layout.addStretch()
        
    def load_wallets(self):
        """Загружает список сохраненных кошельков"""
        try:
            from plugins.web3_wallet import Web3Wallet
            wallets = Web3Wallet.get_saved_wallets()
            
            self.wallet_combo.clear()
            if wallets:
                for wallet in wallets:
                    display_text = f"{wallet['name']} ({wallet['address'][:10]}...{wallet['address'][-6:]})"
                    self.wallet_combo.addItem(display_text, wallet['name'])
            else:
                self.wallet_combo.addItem("Нет сохраненных кошельков", "")
            
            # Обновляем состояние кнопки удаления
            self._on_wallet_selected()
        except Exception as e:
            print(f"Ошибка загрузки кошельков: {e}")
            self.wallet_combo.addItem("Ошибка загрузки кошельков", "")
            self.delete_wallet_btn.setEnabled(False)
    
    def _on_wallet_selected(self):
        """Обработчик выбора кошелька из списка"""
        wallet_name = self.wallet_combo.currentData()
        # Включаем кнопку удаления только если выбран реальный кошелек
        self.delete_wallet_btn.setEnabled(wallet_name is not None and wallet_name != "")
    
    def _delete_selected_wallet(self):
        """Удаляет выбранный кошелек"""
        wallet_name = self.wallet_combo.currentData()
        if not wallet_name:
            QMessageBox.warning(self, "Ошибка", "Выберите кошелек для удаления")
            return
        
        self._delete_wallet(wallet_name)
    
    def _delete_wallet(self, wallet_name):
        """Удаляет кошелек"""
        reply = QMessageBox.question(
            self,
            "Подтверждение удаления",
            f"Вы уверены, что хотите удалить кошелек '{wallet_name}'?\n\nЭто действие нельзя отменить!",
            QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No,
            QMessageBox.StandardButton.No
        )
        
        if reply == QMessageBox.StandardButton.Yes:
            try:
                import os
                wallet_path = os.path.join("config/wallets", f"{wallet_name}.json")
                if os.path.exists(wallet_path):
                    os.remove(wallet_path)
                    print(f"✅ Кошелек '{wallet_name}' удален")
                    
                    # Перезагружаем список кошельков
                    self.load_wallets()
                    
                    QMessageBox.information(self, "Успех", f"Кошелек '{wallet_name}' успешно удален")
                else:
                    QMessageBox.warning(self, "Ошибка", f"Файл кошелька '{wallet_name}' не найден")
            except Exception as e:
                print(f"❌ Ошибка удаления кошелька: {e}")
                QMessageBox.critical(self, "Ошибка", f"Не удалось удалить кошелек: {e}")
        
    def unlock_wallet(self):
        """Войти в кошелек"""
        # Получаем выбранный кошелек из выпадающего списка
        wallet_name = self.wallet_combo.currentData()
        password = self.password_input.text()
        
        if not wallet_name:
            self.show_error("Выберите кошелек из списка")
            return
            
        if not password:
            self.show_error("Введите пароль")
            return
        
        # Отправляем сигнал с названием кошелька и паролем
        self.wallet_unlocked.emit(wallet_name, password)
    
    def show_error(self, message):
        """Показать ошибку"""
        print(f"❌ Ошибка: {message}")
        # Можно добавить QMessageBox для показа ошибки пользователю
        from PyQt6.QtWidgets import QMessageBox
        msg = QMessageBox()
        msg.setIcon(QMessageBox.Icon.Warning)
        msg.setWindowTitle("Ошибка")
        msg.setText(message)
        msg.exec()
