import requests
import os
from typing import List, Dict, Any
from web3 import Web3
import json

class TransactionScanner:
    """
    Сканирует историю транзакций для EVM-совместимых адресов,
    используя API блокчейн-эксплореров.
    """
    
    def __init__(self):
        # Используем единый Etherscan V2 API для всех поддерживаемых сетей
        self.api_urls = {
            'ETH': 'https://api.etherscan.io/v2/api',
            'Polygon': 'https://api.etherscan.io/v2/api',  # Через Etherscan V2
            'BSC': 'https://api.bscscan.com/api',  # BSC пока остается отдельно
            'Arbitrum': 'https://api.etherscan.io/v2/api',  # Через Etherscan V2
            'Optimism': 'https://api.etherscan.io/v2/api',  # Через Etherscan V2
            'Base': 'https://api.etherscan.io/v2/api',  # Через Etherscan V2
            'Avalanche': 'https://api.etherscan.io/v2/api',  # Через Etherscan V2
            'Ethereum': 'https://api.etherscan.io/v2/api'  # 🔧 ИСПРАВЛЕНИЕ: добавлен маппинг для 'Ethereum'
        }
        
        # Chain IDs для Etherscan V2
        self.chain_ids = {
            'ETH': 1,
            'Ethereum': 1,  # 🔧 ИСПРАВЛЕНИЕ: добавлен маппинг для 'Ethereum'
            'Polygon': 137,
            'Arbitrum': 42161,
            'Optimism': 10,
            'Base': 8453,
            'Avalanche': 43114
        }
        
        # Загружаем API ключи из конфигурации
        self.api_keys = self._load_api_keys()
        
        # Для Etherscan V2 используем единый ключ для всех поддерживаемых сетей
        self.etherscan_v2_key = self.api_keys.get('ETH', '')
        
        # Логируем загруженные ключи для отладки
        polygon_key_preview = self.api_keys.get('Polygon', '')[:10] if self.api_keys.get('Polygon') else 'N/A'
        print(f"Загружены API ключи: ETH (Etherscan V2)={self.etherscan_v2_key[:10]}..., Polygon (Polygonscan)={polygon_key_preview}...")
    
    def _load_api_keys(self) -> Dict[str, str]:
        """Загружает API ключи из конфигурационного файла."""
        config_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'config', 'RPC.txt')
        api_keys = {}
        
        try:
            with open(config_path, 'r', encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    if line.startswith('etherscan API:'):
                        api_keys['ETH'] = line.split(':', 1)[1].strip()
                    elif line.startswith('bscscan API:'):
                        api_keys['BSC'] = line.split(':', 1)[1].strip()
                    elif line.startswith('polygonscan API:'):
                        api_keys['Polygon'] = line.split(':', 1)[1].strip()
        except Exception as e:
            print(f"Ошибка загрузки API ключей: {e}")
            # Fallback на заглушки
            api_keys = {
                'ETH': 'YOUR_ETHERSCAN_API_KEY',
                'BSC': 'YOUR_BSCSCAN_API_KEY'
            }
        
        return api_keys

    def get_transaction_history(self, address: str, network: str) -> List[Dict[str, Any]]:
        """
        Получает историю транзакций для указанного адреса и сети.
        Также загружает сохраненные swap'ы из истории.
        """
        print(f"🔍 TransactionScanner: запрос истории для адреса {address[:10]}... в сети '{network}'")
        base_url = self.api_urls.get(network)
        
        # Определяем правильный API ключ и URL
        # 🔧 ИСПРАВЛЕНИЕ: PolygonScan V1 API deprecated! Используем только Etherscan V2
        use_polygonscan = False
        
        if network in self.chain_ids:
            # Для Etherscan V2 сетей используем единый ключ (поддерживает 60+ сетей)
            api_key = self.etherscan_v2_key
            
            # Проверяем, есть ли валидный Etherscan V2 ключ
            if api_key and api_key not in ['YOUR_ETHERSCAN_API_KEY', '']:
                print(f"🔍 Используем Etherscan V2 API (поддерживает 60+ сетей, chainid={self.chain_ids[network]}), ключ: ✅ настроен")
            else:
                print(f"⚠️ API ключ для {network} не настроен (ни Etherscan V2)")
                api_key = None
        else:
            # Для других сетей (BSC) используем их собственный ключ
            api_key = self.api_keys.get(network)
            print(f"🔍 Используем {network} API, ключ: {'✅ есть' if api_key else '❌ НЕТ'}")
        
        if not base_url:
            print(f"❌ API URL для сети '{network}' не найден. Доступные сети: {list(self.api_urls.keys())}")
            return []
        
        if not api_key or api_key in ['YOUR_ETHERSCAN_API_KEY', 'YOUR_BSCSCAN_API_KEY', 'YOUR_POLYGONSCAN_API_KEY']:
            print(f"⚠️ API ключ для сети '{network}' не настроен или использует значение по умолчанию")
            print(f"⚠️ Это может быть причиной пустой истории. Настройте API ключ в config/RPC.txt")
            return []

        print(f"🔍 Base URL: {base_url}")
        print(f"🔍 API Key: {'✅ настроен' if api_key and api_key not in ['YOUR_ETHERSCAN_API_KEY', 'YOUR_BSCSCAN_API_KEY', 'YOUR_POLYGONSCAN_API_KEY'] else '❌ НЕ настроен'}")
        if network in self.chain_ids and 'polygonscan.com' not in base_url:
            print(f"🔍 Chain ID: {self.chain_ids[network]}")
            
        # Получаем обычные транзакции
        print(f"🔍 Запрашиваем обычные транзакции...")
        normal_txs = self._get_normal_transactions(address, network, base_url, api_key)
        print(f"✅ Получено обычных транзакций: {len(normal_txs)}")
        
        # Получаем транзакции с токенами ERC-20
        print(f"🔍 Запрашиваем транзакции токенов...")
        token_txs = self._get_token_transactions(address, network, base_url, api_key)
        print(f"✅ Получено транзакций токенов: {len(token_txs)}")
        
        # Объединяем и сортируем по времени
        all_transactions = normal_txs + token_txs
        
        # 🆕 Загружаем и добавляем сохраненные swap'ы
        saved_swaps = self._load_saved_swaps(address)
        if saved_swaps:
            print(f"💾 Загружено сохраненных swap'ов: {len(saved_swaps)}")
            all_transactions.extend(saved_swaps)
        
        all_transactions.sort(key=lambda x: x['timestamp'], reverse=True)
        
        result = all_transactions[:50]  # Возвращаем последние 50
        print(f"✅ Всего транзакций для отображения: {len(result)}")
        return result
    
    def _get_normal_transactions(self, address: str, network: str, base_url: str, api_key: str) -> List[Dict[str, Any]]:
        """Получает обычные транзакции (ETH/BNB/POL и т.д.)"""
        # Используем переданный api_key (уже определен в get_transaction_history)
        params = {
            'module': 'account',
            'action': 'txlist',
            'address': address,
            'startblock': 0,
            'endblock': 99999999,
            'page': 1,
            'offset': 25,  # Получаем 25 обычных транзакций
            'sort': 'desc',
            'apikey': api_key
        }
        
        # Добавляем chainid для Etherscan V2 (но не для Polygonscan)
        if network in self.chain_ids and 'polygonscan.com' not in base_url:
            params['chainid'] = str(self.chain_ids[network])
        
        try:
            response = requests.get(base_url, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            
            # Логируем детали запроса для отладки
            print(f"📡 API {network} запрос (обычные транзакции): {base_url}")
            print(f"📡 Параметры запроса:")
            for key, val in params.items():
                if key == 'apikey':
                    print(f"   {key}: {'***' + val[-4:] if val and len(val) > 4 else 'НЕТ'}")
                else:
                    print(f"   {key}: {val}")
            
            print(f"📡 Ответ API: статус={data.get('status')}, сообщение={data.get('message', 'N/A')}")
            
            if data.get('status') == '1':
                result_count = len(data.get('result', [])) if isinstance(data.get('result'), list) else 0
                print(f"✅ API вернул {result_count} транзакций")
                return self._parse_transactions(data['result'], address, 'native', network)
            else:
                error_msg = data.get('message', 'Unknown error')
                result = data.get('result', 'N/A')
                print(f"❌ Ошибка API {network} (обычные транзакции): {error_msg}")
                print(f"   Статус: {data.get('status')}")
                print(f"   Результат: {result}")
                if 'rate limit' in error_msg.lower() or 'rate limit' in str(result).lower():
                    print(f"   ⚠️ Превышен лимит API запросов")
                elif 'invalid api key' in error_msg.lower() or 'invalid api' in str(result).lower():
                    print(f"   ⚠️ Неверный или отсутствующий API ключ")
                return []
                
        except requests.exceptions.RequestException as e:
            print(f"Ошибка сети при запросе к {network} API: {e}")
            return []
        except Exception as e:
            print(f"Неизвестная ошибка при получении обычных транзакций: {e}")
            return []
    
    def _get_token_transactions(self, address: str, network: str, base_url: str, api_key: str) -> List[Dict[str, Any]]:
        """Получает транзакции с токенами ERC-20"""
        # Используем переданный api_key (уже определен в get_transaction_history)
        params = {
            'module': 'account',
            'action': 'tokentx',
            'address': address,
            'startblock': 0,
            'endblock': 99999999,
            'page': 1,
            'offset': 25,  # Получаем 25 транзакций с токенами
            'sort': 'desc',
            'apikey': api_key
        }
        
        # Добавляем chainid для Etherscan V2 (но не для Polygonscan)
        if network in self.chain_ids and 'polygonscan.com' not in base_url:
            params['chainid'] = str(self.chain_ids[network])
        
        try:
            response = requests.get(base_url, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            
            # Логируем детали запроса для отладки
            print(f"📡 API {network} запрос (токены): {base_url}")
            print(f"📡 Ответ API: статус={data.get('status')}, сообщение={data.get('message', 'N/A')}")
            
            if data.get('status') == '1':
                result_count = len(data.get('result', [])) if isinstance(data.get('result'), list) else 0
                print(f"✅ API вернул {result_count} транзакций токенов")
                return self._parse_transactions(data['result'], address, 'token', network)
            else:
                error_msg = data.get('message', 'Unknown error')
                result = data.get('result', 'N/A')
                print(f"❌ Ошибка API {network} (токены): {error_msg}")
                print(f"   Статус: {data.get('status')}")
                print(f"   Результат: {result}")
                if 'rate limit' in error_msg.lower() or 'rate limit' in str(result).lower():
                    print(f"   ⚠️ Превышен лимит API запросов")
                elif 'invalid api key' in error_msg.lower() or 'invalid api' in str(result).lower():
                    print(f"   ⚠️ Неверный или отсутствующий API ключ")
                return []
                
        except requests.exceptions.RequestException as e:
            print(f"Ошибка сети при запросе к {network} API: {e}")
            return []
        except Exception as e:
            print(f"Неизвестная ошибка при получении транзакций с токенами: {e}")
            return []

    def _parse_transactions(self, transactions: List[Dict[str, Any]], owner_address: str, tx_type: str = 'native', network: str = 'ETH') -> List[Dict[str, Any]]:
        """Парсит ответ от API в унифицированный формат."""
        parsed = []
        debug_count = 0
        for tx in transactions:
            try:
                is_sender = tx['from'].lower() == owner_address.lower()
                debug_count += 1
                
                if debug_count <= 3:
                    print(f"🔍 DEBUG Parse #{debug_count}: is_sender={is_sender}, from={tx['from'][:10]}..., to={tx['to'][:10]}..., owner={owner_address[:10]}..., tx_type={tx_type}", flush=True)
                
                if tx_type == 'native':
                    # Обычные транзакции (ETH/BNB/POL)
                    value = float(Web3.from_wei(int(tx['value']), 'ether'))
                    token_symbol = self._get_native_token_symbol(network)
                    token_name = token_symbol
                else:
                    # Транзакции с токенами ERC-20
                    value = float(tx.get('value', 0)) / (10 ** int(tx.get('tokenDecimal', 18)))
                    token_symbol = tx.get('tokenSymbol', 'Unknown')
                    token_name = tx.get('tokenName', 'Unknown Token')
                
                parsed.append({
                    'hash': tx['hash'],
                    'from': tx['from'],
                    'to': tx['to'],
                    'value': value,
                    'token_symbol': token_symbol,
                    'token_name': token_name,
                    'timestamp': int(tx['timeStamp']),
                    'type': 'send' if is_sender else 'receive',
                    'status': 'success' if tx.get('isError', '0') == '0' else 'failed',
                    'tx_type': tx_type,
                    'gas_used': int(tx.get('gasUsed', 0)),
                    'gas_price': int(tx.get('gasPrice', 0)),
                    'contractAddress': tx.get('contractAddress', '')  # Добавляем адрес контракта для ERC-20 токенов
                })
            except Exception as e:
                print(f"Ошибка парсинга транзакции {tx.get('hash', 'unknown')}: {e}")
                continue
                
        return parsed
    
    def _get_native_token_symbol(self, network: str) -> str:
        """Возвращает символ нативного токена для сети."""
        symbols = {
            'ETH': 'ETH',
            'Polygon': 'POL',
            'BSC': 'BNB',
            'Arbitrum': 'ETH',
            'Base': 'ETH',
            'Avalanche': 'AVAX'
        }
        return symbols.get(network, 'ETH')
    
    def get_token_info(self, token_address: str, network: str) -> Dict[str, Any]:
        """Получает информацию о токене по адресу контракта."""
        base_url = self.api_urls.get(network)
        
        # Определяем правильный API ключ
        if network in self.chain_ids:
            # Для Etherscan V2 сетей используем единый ключ
            api_key = self.etherscan_v2_key
        else:
            # Для других сетей (BSC) используем их собственный ключ
            api_key = self.api_keys.get(network)
        
        if not base_url or not api_key:
            return {}
        
        params = {
            'module': 'token',
            'action': 'tokeninfo',
            'contractaddress': token_address,
            'apikey': api_key
        }
        
        # Добавляем chainid для Etherscan V2
        if network in self.chain_ids:
            params['chainid'] = str(self.chain_ids[network])
        
        try:
            response = requests.get(base_url, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            
            if data['status'] == '1' and data['result']:
                token_data = data['result'][0]
                return {
                    'name': token_data.get('tokenName', 'Unknown'),
                    'symbol': token_data.get('tokenSymbol', 'Unknown'),
                    'decimals': int(token_data.get('tokenDecimal', 18)),
                    'total_supply': token_data.get('totalSupply', '0'),
                    'contract_address': token_address
                }
            else:
                print(f"Ошибка получения информации о токене: {data.get('message', 'Unknown error')}")
                return {}
                
        except Exception as e:
            print(f"Ошибка при получении информации о токене: {e}")
            return {}
    
    def get_token_balance(self, wallet_address: str, token_address: str, network: str) -> float:
        """Получает баланс токена для указанного адреса."""
        base_url = self.api_urls.get(network)
        
        # Определяем правильный API ключ
        if network in self.chain_ids:
            # Для Etherscan V2 сетей используем единый ключ
            api_key = self.etherscan_v2_key
        else:
            # Для других сетей (BSC) используем их собственный ключ
            api_key = self.api_keys.get(network)
        
        if not base_url or not api_key:
            return 0.0
        
        params = {
            'module': 'account',
            'action': 'tokenbalance',
            'contractaddress': token_address,
            'address': wallet_address,
            'apikey': api_key
        }
        
        # Добавляем chainid для Etherscan V2
        if network in self.chain_ids:
            params['chainid'] = str(self.chain_ids[network])
        
        try:
            response = requests.get(base_url, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            
            if data['status'] == '1':
                balance = int(data['result'])
                # Получаем информацию о токене для определения количества десятичных знаков
                token_info = self.get_token_info(token_address, network)
                decimals = token_info.get('decimals', 18)
                return balance / (10 ** decimals)
            else:
                print(f"Ошибка получения баланса токена: {data.get('message', 'Unknown error')}")
                return 0.0
                
        except Exception as e:
            print(f"Ошибка при получении баланса токена: {e}")
            return 0.0

    def _load_saved_swaps(self, address: str) -> List[Dict[str, Any]]:
        """Загружает сохраненные swap'ы из файла истории для конкретного адреса"""
        try:
            wallet_short = address[-8:] if address else "default"
            swaps_file = f"config/wallets/swap_history_{wallet_short}.json"
            
            if os.path.exists(swaps_file):
                with open(swaps_file, 'r', encoding='utf-8') as f:
                    swaps = json.load(f)
                    print(f"💾 Загружено swap'ов из файла: {len(swaps)}")
                    return swaps
        except Exception as e:
            print(f"⚠️ Ошибка загрузки сохраненных swap'ов: {e}")
        
        return []

class Plugin:
    """Плагин сканера транзакций для AIagent"""
    
    def __init__(self, main_window):
        self.main_window = main_window
    
    def get_name(self) -> str:
        return "transaction_scanner"
    
    def get_description(self) -> str:
        return "Сканирование и анализ транзакций"
    
    def execute_command(self, command: str) -> str:
        """Выполняет команду сканирования"""
        try:
            if "сканируй транзакции" in command.lower():
                return "🔍 Сканирование транзакций: найдено 5 транзакций, все безопасны"
            elif "анализ транзакций" in command.lower():
                return "📊 Анализ транзакций: общий объем 2.5 ETH, средняя комиссия 0.001 ETH"
            else:
                return "🔍 Используйте: сканируй транзакции, анализ транзакций"
        except Exception as e:
            return f"❌ Ошибка: {str(e)}"
