#!/usr/bin/env python3
"""
Сервис для получения реальных цен токенов
Использует бесплатные API: CoinGecko, CoinMarketCap (если есть ключ)
"""

import requests
import json
import time
from typing import Dict, List, Optional, Any
from decimal import Decimal
import os
from config.settings import settings_manager

# Глобальная переменная для сигнала превышения лимита API
api_rate_limit_signal = None

def set_api_rate_limit_signal(signal):
    """Устанавливает сигнал для уведомления о превышении лимита API"""
    global api_rate_limit_signal
    api_rate_limit_signal = signal

class TokenPriceService:
    """
    Сервис для получения актуальных цен токенов
    """
    
    def __init__(self):
        self.session = requests.Session()
        self.session.headers.update({
            'User-Agent': 'MaxportAIagent-Wallet/1.0',
            'Accept': 'application/json'
        })
        
        # Загружаем конфигурацию
        self.config = self._load_config()
        
        # API endpoints
        self.apis = {
            'coingecko': {
                'base_url': 'https://api.coingecko.com/api/v3',
                'free': True,
                'rate_limit': 50,  # запросов в минуту
                'last_request': 0,
                'min_interval': 3.0,  # увеличиваем интервал для избежания 429 ошибок
                'max_tokens_per_request': 100  # уменьшаем для стабильности
            },
            'coinmarketcap': {
                'base_url': 'https://pro-api.coinmarketcap.com/v1',
                'free': False,
                'api_key': self.config.get('COINMARKETCAP_API_KEY'),
                'rate_limit': 10000,  # запросов в месяц для бесплатного плана
                'last_request': 0
            },
            'uniswap': {
                'base_url': 'https://api.uniswap.org/v1',
                'free': True,
                'rate_limit': 1000,  # запросов в минуту
                'last_request': 0,
                'min_interval': 0.1
            },
            '1inch': {
                'base_url': 'https://api.1inch.dev',
                'free': True,
                'rate_limit': 1000,  # запросов в минуту
                'last_request': 0,
                'min_interval': 0.1
            },
            'chainlink': {
                'free': True,
                'rate_limit': 1000,
                'last_request': 0,
                'min_interval': 0.1
            },
            '0x': {
                'base_url': 'https://api.0x.org',
                'free': True,
                'rate_limit': 1000,  # запросов в минуту
                'last_request': 0,
                'min_interval': 0.1,
                'api_key': 'd34afb90-4beb-45d5-8a86-ec0f5f61fd3f'
            }
        }
        
        # Кэш цен (время жизни 5 минут)
        self.price_cache = {}
        self.cache_ttl = 300  # 5 минут
        
        # Глобальный кэш для всех экземпляров
        self._global_cache = {}
        self._global_cache_ttl = 300  # 5 минут
        
        # Маппинг символов токенов к CoinGecko ID
        self.symbol_to_id = {
            'ETH': 'ethereum',
            'BTC': 'bitcoin',
            'USDT': 'tether',
            'USDC': 'usd-coin',
            'DAI': 'dai',
            'BNB': 'binancecoin',
            'POL': 'matic-network',
            'SOL': 'solana',
            'XRP': 'ripple',
            'ADA': 'cardano',
            'AVAX': 'avalanche-2',
            'DOGE': 'dogecoin',
            'TRX': 'tron',
            'DOT': 'polkadot',
            'LINK': 'chainlink',
            'TON': 'the-open-network',
            'SHIB': 'shiba-inu',
            'LTC': 'litecoin',
            'BCH': 'bitcoin-cash',
            'ICP': 'internet-computer',
            'UNI': 'uniswap',
            'NEAR': 'near',
            'KAS': 'kaspa',
            'LEO': 'leo-token',
            'OKB': 'okb',
            'XLM': 'stellar',
            'ETC': 'ethereum-classic',
            'XMR': 'monero',
            'APT': 'aptos',
            'HBAR': 'hedera-hashgraph',
            'CRO': 'crypto-com-chain',
            'VET': 'vechain',
            'FIL': 'filecoin',
            'ATOM': 'cosmos',
            'ARB': 'arbitrum',
            'RNDR': 'render-token',
            'IMX': 'immutable-x',
            'GRT': 'the-graph',
            'OP': 'optimism',
            'INJ': 'injective',
            'THETA': 'theta-token',
            'XTZ': 'tezos',
            'SEI': 'sei-network',
            'XAS': 'asd',
            'FTM': 'fantom',
            'LDO': 'lido-dao',
            'TIA': 'celestia',
            'BONK': 'bonk',
            'WBTC': 'wrapped-bitcoin',
            'STX': 'stacks',
            'SUI': 'sui',
            'EGLD': 'multiversx',
            'AXS': 'axie-infinity',
            'SAND': 'the-sandbox',
            'MANA': 'decentraland',
            'FLOW': 'flow',
            'APE': 'apecoin',
            'CHZ': 'chiliz',
            'MINA': 'mina-protocol',
            'GALA': 'gala',
            'MKR': 'maker',
            'SNX': 'havven',
            'AAVE': 'aave',
            'COMP': 'compound-governance-token',
            'YFI': 'yearn-finance',
            'CRV': 'curve-dao-token',
            'BAL': 'balancer',
            '1INCH': '1inch',
            'ZRX': '0x',
            'KAVA': 'kava',
            'SUSHI': 'sushi',
            'RUNE': 'thorchain',
            'LRC': 'loopring',
            'ENS': 'ethereum-name-service',
            'DYDX': 'dydx',
            'GMX': 'gmx',
            'CAKE': 'pancakeswap-token',
            'TWT': 'trust-wallet-token',
            'CVX': 'convex-finance',
            'FXS': 'frax-share',
            'OCEAN': 'ocean-protocol',
            'FET': 'fetch-ai',
            'AGIX': 'singularitynet',
            'ROSE': 'oasis-network',
            'CELO': 'celo',
            'SKL': 'skale',
            'RLC': 'iexec-rlc',
            'BAND': 'band-protocol',
            'ANKR': 'ankr',
            'OMI': 'ecomi',
            'STORJ': 'storj',
            'UMA': 'uma',
            'POLY': 'polymath',
            'COTI': 'coti',
            'POLS': 'polkastarter',
            'GLM': 'golem',
            'PAXG': 'pax-gold',
            'USDP': 'paxos-standard',
            'GUSD': 'gemini-dollar',
            'BUSD': 'binance-usd'
        }
        
        # Обратный маппинг для поиска по символу
        self.symbol_to_id = {symbol.upper(): coin_id for symbol, coin_id in self.symbol_to_id.items()}
    
    def _load_config(self) -> Dict[str, str]:
        """Загружает конфигурацию из файла config/RPC.txt."""
        config = {}
        try:
            config_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'config', 'RPC.txt')
            with open(config_path, "r", encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    if not line or line.startswith('#'):
                        continue
                    if '=' in line:
                        key, value = line.split('=', 1)
                        config[key.strip()] = value.strip()
        except Exception as e:
            print(f"Ошибка загрузки конфигурации: {e}")
        return config
    
    def _check_rate_limit(self, api_name: str) -> bool:
        """Проверяет лимит запросов для API"""
        api = self.apis.get(api_name)
        if not api:
            return False
            
        current_time = time.time()
        
        # Для CoinGecko: минимальный интервал между запросами
        if api_name == 'coingecko':
            if current_time - api['last_request'] < api.get('min_interval', 1.2):
                return False
            # Не обновляем last_request здесь, только при успешном запросе
            
        # Для CoinMarketCap: проверяем ключ
        elif api_name == 'coinmarketcap':
            if not api['api_key'] or api['api_key'] == 'YOUR_COINMARKETCAP_API_KEY':
                return False
                
        return True
    
    def get_token_price(self, token_symbol: str, currency: str = 'usd', force_refresh: bool = False) -> Optional[float]:
        """
        Получает текущую цену токена
        
        Args:
            token_symbol: Символ токена (например, 'ETH', 'BTC')
            currency: Валюта для цены (например, 'usd', 'eur', 'rub')
            force_refresh: Принудительно обновить цену, игнорируя кэш
            
        Returns:
            Цена токена в указанной валюте или None при ошибке
        """
        # Проверяем кэш (если не принудительное обновление)
        if not force_refresh:
            cache_key = f"{token_symbol.upper()}_{currency.lower()}"
            if cache_key in self.price_cache:
                cached_price, timestamp = self.price_cache[cache_key]
                if time.time() - timestamp < self.cache_ttl:
                    return cached_price
        
        # Пытаемся получить цену через CoinGecko (бесплатно)
        price = self._get_coingecko_price(token_symbol, currency)
        if price is not None:
            # Сохраняем в кэш
            cache_key = f"{token_symbol.upper()}_{currency.lower()}"
            self.price_cache[cache_key] = (price, time.time())
            return price
        
        # Если CoinGecko не сработал, пробуем альтернативные источники
        print(f"🔄 CoinGecko недоступен, пробуем альтернативные источники для {token_symbol}...")
        
        # Пробуем Uniswap
        price = self._get_uniswap_price(token_symbol, currency)
        if price is not None:
            cache_key = f"{token_symbol.upper()}_{currency.lower()}"
            self.price_cache[cache_key] = (price, time.time())
            return price
        
        # Пробуем 1inch
        price = self._get_1inch_price(token_symbol, currency)
        if price is not None:
            cache_key = f"{token_symbol.upper()}_{currency.lower()}"
            self.price_cache[cache_key] = (price, time.time())
            return price
        
        # Пробуем 0x API
        price = self._get_0x_price(token_symbol, currency)
        if price is not None:
            cache_key = f"{token_symbol.upper()}_{currency.lower()}"
            self.price_cache[cache_key] = (price, time.time())
            return price
        
        # Пробуем CoinMarketCap (если есть ключ)
        if self.apis['coinmarketcap']['api_key'] and self.apis['coinmarketcap']['api_key'] != 'YOUR_COINMARKETCAP_API_KEY':
            price = self._get_coinmarketcap_price(token_symbol, currency)
            if price is not None:
                cache_key = f"{token_symbol.upper()}_{currency.lower()}"
                self.price_cache[cache_key] = (price, time.time())
                return price
        
        # Fallback: возвращаем примерные цены для основных токенов
        fallback_prices = self._get_fallback_price(token_symbol, currency)
        if fallback_prices is not None:
            # Сохраняем в кэш с коротким TTL
            cache_key = f"{token_symbol.upper()}_{currency.lower()}"
            self.price_cache[cache_key] = (fallback_prices, time.time())
            return fallback_prices
        
        return None
    
    def _get_coingecko_price(self, token_symbol: str, currency: str = 'usd') -> Optional[float]:
        """Получает цену токена через CoinGecko API"""
        try:
            if not self._check_rate_limit('coingecko'):
                print(f"Rate limit для CoinGecko превышен")
                return None
            
            # Получаем CoinGecko ID для токена
            coin_id = self.symbol_to_id.get(token_symbol.upper())
            if not coin_id:
                print(f"Токен {token_symbol} не найден в маппинге CoinGecko")
                return None
            
            # Формируем URL для запроса
            url = f"{self.apis['coingecko']['base_url']}/simple/price"
            params = {
                'ids': coin_id,
                'vs_currencies': currency.lower()
            }
            
            response = self.session.get(url, params=params, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                
                if coin_id in data and currency.lower() in data[coin_id]:
                    price = data[coin_id][currency.lower()]
                    # Обновляем last_request только при успешном запросе
                    self.apis['coingecko']['last_request'] = time.time()
                    return price
                else:
                    print(f"Цена для {token_symbol} не найдена в ответе CoinGecko")
                    return None
            elif response.status_code == 429:
                print(f"CoinGecko API: Превышен лимит запросов (429)")
                # Уведомляем о превышении лимита API
                if api_rate_limit_signal:
                    api_rate_limit_signal.emit("CoinGecko", "Превышен лимит запросов (429)")
                return None
            else:
                print(f"Ошибка CoinGecko API: {response.status_code}")
                return None
                
        except Exception as e:
            print(f"❌ Ошибка получения цены через CoinGecko: {e}")
            import traceback
            traceback.print_exc()
            return None
    
    def _get_token_address(self, token_symbol: str) -> Optional[str]:
        """Получает адрес токена по символу"""
        # Маппинг основных токенов к их адресам (Ethereum Mainnet)
        token_addresses = {
            'ETH': '0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2',  # WETH
            'WETH': '0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2',
            'USDC': '0xA0b86a33E6441b8c4C8C1C1e0F0B2e639B1717e1',
            'USDT': '0xdAC17F958D2ee523a2206206994597C13D831ec7',
            'DAI': '0x6B175474E89094C44Da98b954EedeAC495271d0F',
            'WBTC': '0x2260FAC5E5542a773Aa44fBCfeDf7C193bc2C599',
            'BTC': '0x2260FAC5E5542a773Aa44fBCfeDf7C193bc2C599',  # WBTC как BTC
            'UNI': '0x1f9840a85d5aF5bf1D1762F925BDADdC4201F984',
            'LINK': '0x514910771AF9Ca656af840dff83E8264EcF986CA',
            'POL': '0x7D1AfA7B718fb893dB30A3aBc0Cfc608aC1fEce8',
            'AAVE': '0x7Fc66500c84A76Ad7e9c93437bFc5Ac33E2DDaE9',
            'CRV': '0xD533a949740bb3306d119CC777fa900bA034cd52',
            'SNX': '0xC011a73ee8576Fb46F5E1c5751cA3B9Fe0af2a6F',
            'COMP': '0xc00e94Cb662C3520282E6f5717214004A7f26888',
            'YFI': '0x0bc529c0C6419Bc4D2C45Dc8eC9EdD6057358192',
            'SUSHI': '0x6B3595068778DD592e39A122f4f5a5cF09C90fE2',
            'BAL': '0xba100000625a3754423978a60c9317c58a424e3D',
            'REN': '0x408e41876cCdcDCfF10b4b4e9B162Bd8E5fB70f5',
            'BAND': '0xBA11D00c5f74255f56a5E366F4F77f5A186d7f55',
            'ZRX': '0xE41d2489571d322189246DaFA5ebDe1F4699F498',
            'KNC': '0xdd974D5C2e2928deA5F71b9825b8b646686BD200',
            'MANA': '0x0F5D2fB29fb7d3CFeE444a200298f468908cC942',
            'ENJ': '0xF629cBd94d3791C9250152BD8dfBDF380E2a3B9c',
            'REP': '0x1985365e9f78359a9B6AD760e32412f4a445E862',
            'BAT': '0x0D8775F648430679A709E98d2b0Cb6250d2887EF',
            'ZEN': '0x2bD9a9553e0f7F4c8C9D8B9eb9D8B8D8B8D8B8D',
            'OMG': '0xd26114cd6EE289AccF82350c8d8487fedB8A0C07',
            'GNT': '0xa74476443119A942dE498590Fe1f2454d7D4aC0d',
            'STORJ': '0xB64ef51C888972c908CFacf59B47C1AfBC0Ab8aC',
            'FUN': '0x419D0d8BdD9aF5e606Ae2232ed285Aff190E711b',
            'PAY': '0xEa5f88E54d982Cbb0c441cde4E79bC305e5b43Bc',
            'RLC': '0x607F4C5BB672230e8672085532f7e901544a7375',
            'RCN': '0xF970b8E36e23F7Fc3FD508Ee9A1f1710d9E8C6b6',
            'MKR': '0x9f8F72aA9304c8B593d555F12eF6589cC3A579A2',
            'MLN': '0xec67005c4E498Ec7f55E092bd1d35cbC47C91892',
            'REQ': '0x8f8221aFbB33998d8584A2B05749bA73c37a938a',
            'DNT': '0x0AbdAce70D3790235af448C88547603b945604ea',
            'TRX': '0xf230b790E05390FC8295F4d3F60332c93Bd42e51',
            'DOT': '0x43Dfc4159D86F3A37A5A4B3D4580b7a7Bc1a6f4c',
            'ADA': '0x3EE2200Efb3400fAbB9AacF31297cBdD1d435D47',
            'XRP': '0x39fBBABf11738317a448031930706cd3e612e7bA',
            'LTC': '0x2A3bFF78B79A009976EeA096a51A948a3dC00e34',
            'BCH': '0x9C58BAcB3319b3562c9Dc58DD4c4E411f8Ac56e5',
            'EOS': '0x86Fa049857E0209aa7D9e616F7eb3b3B78ECfdb0',
            'XLM': '0x056Fd409E1d7A124BD7017459dFEa2F387b6d5Cd',
            'XTZ': '0x2C71b0d8D8Bd6BDE2C2D8fc90067dC5e0c413dC26',
            'VET': '0x6FDcdfef7c496407cCb0cEC90f07C63C1417b8D5',
            'ICX': '0xb5A4c7d4847E4b4A9918479d6D30A2Ec9b771501',
            'WAN': '0x39Bb259F66E1C59d5ACeC9B5707C7A0c2D427602',
            'AION': '0x4CEdA7906a5Ed2179785Cd3A40A69ee8bc99C466',
            'THETA': '0x3883f5e181fccaF8410FA61e12b59BAd963fb645',
            'QTUM': '0x9a642d6b3368ddc662CA244bAdf32cda716005BC',
            'NEO': '0x8c23B1966030D9964f1066Ee4Ef0083C7fbd6cE7',
            'ONT': '0x4d7C363DED4B3b4e1F954494d2Bc3955eA7f3a1E',
            'IOTA': '0x6fB3e0A217407EFFf7Ca062D46c26E5d60a14d69',
            'XMR': '0x4e5f2F51359Cd2A4C2E4F0d73Ae5732E4F6C6E8B',
            'DASH': '0x6B175474E89094C44Da98b954EedeAC495271d0F',
            'ZEC': '0x4e5f2F51359Cd2A4C2E4F0d73Ae5732E4F6C6E8B'
        }
        return token_addresses.get(token_symbol.upper())
    
    def _get_uniswap_price(self, token_symbol: str, currency: str = 'usd') -> Optional[float]:
        """Получает цену токена через Uniswap API (упрощенная версия)"""
        try:
            if not self._check_rate_limit('uniswap'):
                return None
            
            # Получаем адрес токена
            token_address = self._get_token_address(token_symbol)
            if not token_address:
                return None
            
            # Для демонстрации используем упрощенный подход
            # В реальности нужно вызывать smart contract или использовать Graph API
            print(f"🔍 Пытаемся получить цену {token_symbol} через Uniswap...")
            
            # Возвращаем None для демонстрации - в реальности здесь будет API вызов
            return None
            
        except Exception as e:
            print(f"❌ Ошибка получения цены через Uniswap: {e}")
            return None
    
    def _get_1inch_price(self, token_symbol: str, currency: str = 'usd') -> Optional[float]:
        """Получает цену токена через 1inch API (упрощенная версия)"""
        try:
            if not self._check_rate_limit('1inch'):
                return None
            
            # Получаем адрес токена
            token_address = self._get_token_address(token_symbol)
            if not token_address:
                return None
            
            print(f"🔍 Пытаемся получить цену {token_symbol} через 1inch...")
            
            # Возвращаем None для демонстрации - в реальности здесь будет API вызов
            return None
            
        except Exception as e:
            print(f"❌ Ошибка получения цены через 1inch: {e}")
            return None
    
    def _get_0x_price(self, token_symbol: str, currency: str = 'usd', chain_id: int = 137) -> Optional[float]:
        """Получает цену токена через 0x API v2"""
        try:
            if not self._check_rate_limit('0x'):
                return None
            
            # Импортируем ZeroXV2Integration для получения адресов токенов
            try:
                from plugins.zerox_v2_integration import ZeroXV2Integration
                fee_cfg = settings_manager.get_zerox_fee_config(None) if settings_manager else {}
                zerox_api = ZeroXV2Integration(
                    self.apis['0x']['api_key'],
                    fee_recipient=(fee_cfg or {}).get("recipient") or None,
                    buy_token_percentage_fee=float((fee_cfg or {}).get("buy_token_percentage", 0) or 0.0),
                    sell_token_percentage_fee=float((fee_cfg or {}).get("sell_token_percentage", 0) or 0.0),
                    affiliate_address=(fee_cfg or {}).get("affiliate_address") or None
                )
                
                # Получаем адрес токена для нужной сети
                token_address = zerox_api._get_token_address(token_symbol, chain_id)
                if not token_address:
                    print(f"⚠️ Токен {token_symbol} не найден для сети {chain_id}")
                    return None
                
                # Выбираем базовый токен (USDC) для сети
                base_tokens = {
                    137: "0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174",  # USDC на Polygon (6 decimals)
                    1: "0xA0b86991c6218b36c1d19D4a2e9Eb0cE3606eB48",  # USDC на Ethereum (6 decimals)
                }
                base_token = base_tokens.get(chain_id, base_tokens[137])
                base_decimals = 6  # USDC всегда 6 decimals
                
                # Пропускаем бессмысленные запросы (USDC -> USDC, USDT -> USDT)
                if token_symbol.upper() in ['USDC', 'USDT'] and token_address.lower() == base_token.lower():
                    print(f"⏭️ Пропускаем запрос {token_symbol} -> USDC (один и тот же токен)")
                    return 1.0  # USDC/USDT всегда $1
                
                # Получаем decimals токена
                sell_decimals = zerox_api._get_token_decimals(token_symbol, chain_id)
                
                # Продаем 1 токен, покупаем USDC
                sell_amount = 1.0  # 1 токен
                sell_amount_units = int(sell_amount * (10**sell_decimals))
                
                url = f"{self.apis['0x']['base_url']}/swap/allowance-holder/price"
                params = {
                    'chainId': str(chain_id),
                    'sellToken': token_address,
                    'buyToken': base_token,
                    'sellAmount': str(sell_amount_units),
                    'slippageBps': '100'  # 1% slippage
                }
                
                headers = {
                    '0x-api-key': self.apis['0x']['api_key'],
                    '0x-version': 'v2',
                    'Content-Type': 'application/json'
                }
                
                print(f"🔍 Получаем цену {token_symbol} через 0x API v2 (chainId={chain_id})...")
                
                response = self.session.get(url, params=params, headers=headers, timeout=10)
                
                if response.status_code == 200:
                    data = response.json()
                    
                    # Проверяем наличие нужных полей
                    if 'buyAmount' in data and 'sellAmount' in data:
                        # buyAmount в минимальных единицах USDC (6 decimals)
                        buy_amount_raw = data['buyAmount']
                        if isinstance(buy_amount_raw, str):
                            buy_amount_units = int(buy_amount_raw)
                        else:
                            buy_amount_units = int(buy_amount_raw)
                        buy_amount_usdc = buy_amount_units / (10**base_decimals)
                        
                        # sellAmount в минимальных единицах токена
                        sell_amount_raw = data['sellAmount']
                        if isinstance(sell_amount_raw, str):
                            sell_amount_units_resp = int(sell_amount_raw)
                        else:
                            sell_amount_units_resp = int(sell_amount_raw)
                        sell_amount_tokens = sell_amount_units_resp / (10**sell_decimals)
                        
                        # Цена в USD (USDC = $1)
                        if sell_amount_tokens > 0:
                            price = buy_amount_usdc / sell_amount_tokens
                            
                            self.apis['0x']['last_request'] = time.time()
                            print(f"✅ 0x API: {token_symbol} = ${price:.6f} (1 {token_symbol} = {buy_amount_usdc:.6f} USDC)")
                            return price
                        else:
                            print(f"⚠️ Неверный sellAmount в ответе 0x API")
                            return None
                    else:
                        available_keys = list(data.keys())[:10]
                        print(f"⚠️ Неожиданный формат ответа 0x API: доступные ключи: {available_keys}")
                        
                        # Пробуем получить из tokenMetadata если доступно
                        if 'tokenMetadata' in data:
                            print(f"📋 tokenMetadata: {data.get('tokenMetadata')}")
                        
                        return None
                elif response.status_code == 429:
                    print(f"0x API: Превышен лимит запросов (429)")
                    return None
                else:
                    print(f"⚠️ Ошибка 0x API: {response.status_code}")
                    if response.text:
                        print(f"   Детали: {response.text[:200]}")
                    return None
                    
            except ImportError:
                print(f"⚠️ Не удалось импортировать ZeroXV2Integration")
                return None
                
        except Exception as e:
            print(f"❌ Ошибка получения цены через 0x API: {e}")
            import traceback
            traceback.print_exc()
            return None
    
    def _get_coinmarketcap_price(self, token_symbol: str, currency: str = 'usd') -> Optional[float]:
        """Получает цену токена через CoinMarketCap API"""
        try:
            if not self._check_rate_limit('coinmarketcap'):
                return None
            
            url = f"{self.apis['coinmarketcap']['base_url']}/cryptocurrency/quotes/latest"
            params = {
                'symbol': token_symbol.upper(),
                'convert': currency.upper()
            }
            
            headers = {
                'X-CMC_PRO_API_KEY': self.apis['coinmarketcap']['api_key']
            }
            
            response = self.session.get(url, params=params, headers=headers, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                if 'data' in data and token_symbol.upper() in data['data']:
                    token_data = data['data'][token_symbol.upper()]
                    if 'quote' in token_data and currency.upper() in token_data['quote']:
                        return token_data['quote'][currency.upper()]['price']
                return None
            elif response.status_code == 429:
                print(f"CoinMarketCap API: Превышен лимит запросов (429)")
                # Уведомляем о превышении лимита API
                if api_rate_limit_signal:
                    api_rate_limit_signal.emit("CoinMarketCap", "Превышен лимит запросов (429)")
                return None
            else:
                print(f"Ошибка CoinMarketCap API: {response.status_code}")
                return None
                
        except Exception as e:
            print(f"Ошибка получения цены через CoinMarketCap: {e}")
            return None
    
    def get_multiple_prices(self, token_symbols: List[str], currency: str = 'usd', force_refresh: bool = False) -> Dict[str, float]:
        """
        Получает цены для нескольких токенов одновременно
        
        Args:
            token_symbols: Список символов токенов
            currency: Валюта для цен
            
        Returns:
            Словарь {символ_токена: цена}
        """
        prices = {}
        
        # Сначала проверяем кэш для всех токенов (если не принудительное обновление)
        if not force_refresh:
            cached_count = 0
            for symbol in token_symbols:
                cache_key = f"{symbol.upper()}_{currency.lower()}"
                if cache_key in self.price_cache:
                    cached_price, timestamp = self.price_cache[cache_key]
                    if time.time() - timestamp < self.cache_ttl:
                        prices[symbol] = cached_price
                        cached_count += 1
            
            # Если все цены в кэше, возвращаем их
            if cached_count == len(token_symbols):
                return prices
        
        # Получаем недостающие цены через API
        if force_refresh:
            missing_symbols = token_symbols
        else:
            missing_symbols = [s for s in token_symbols if s not in prices]
        
        # Пытаемся получить цены для недостающих токенов через CoinGecko
        try:
            if missing_symbols and self._check_rate_limit('coingecko'):
                # Получаем CoinGecko ID для недостающих токенов
                coin_ids = []
                symbol_to_id = {}
                
                for symbol in missing_symbols:
                    coin_id = self.symbol_to_id.get(symbol.upper())
                    if coin_id:
                        coin_ids.append(coin_id)
                        symbol_to_id[coin_id] = symbol.upper()
                
                if coin_ids:
                    # Ограничиваем количество токенов в одном запросе
                    max_tokens = self.apis['coingecko']['max_tokens_per_request']
                    if len(coin_ids) > max_tokens:
                        coin_ids = coin_ids[:max_tokens]
                    
                    # Формируем URL для запроса недостающих токенов
                    url = f"{self.apis['coingecko']['base_url']}/simple/price"
                    params = {
                        'ids': ','.join(coin_ids),
                        'vs_currencies': currency.lower()
                    }
                    
                    response = self.session.get(url, params=params, timeout=15)
                    
                    if response.status_code == 200:
                        data = response.json()
                        # Обновляем last_request только при успешном запросе
                        self.apis['coingecko']['last_request'] = time.time()
                        
                        for coin_id, price_data in data.items():
                            if currency.lower() in price_data:
                                symbol = symbol_to_id.get(coin_id)
                                if symbol:
                                    prices[symbol] = price_data[currency.lower()]
                                    # Сохраняем в кэш
                                    cache_key = f"{symbol}_{currency.lower()}"
                                    self.price_cache[cache_key] = (prices[symbol], time.time())
                    elif response.status_code == 429:
                        print("CoinGecko rate limit превышен, используем fallback цены")
                    else:
                        print(f"CoinGecko API error: {response.status_code}")
        except Exception as e:
            print(f"Ошибка получения множественных цен: {e}")
        
        # Fallback: получаем цены по одному
        for symbol in token_symbols:
            price = self.get_token_price(symbol, currency)
            if price is not None:
                prices[symbol] = price
        
        return prices
    
    def get_token_info(self, token_symbol: str) -> Optional[Dict[str, Any]]:
        """
        Получает подробную информацию о токене
        
        Args:
            token_symbol: Символ токена
            
        Returns:
            Словарь с информацией о токене или None
        """
        try:
            if not self._check_rate_limit('coingecko'):
                return None
            
            coin_id = self.symbol_to_id.get(token_symbol.upper())
            if not coin_id:
                return None
            
            url = f"{self.apis['coingecko']['base_url']}/coins/{coin_id}"
            params = {
                'localization': 'false',
                'tickers': 'false',
                'market_data': 'true',
                'community_data': 'false',
                'developer_data': 'false'
            }
            
            response = self.session.get(url, params=params, timeout=10)
            
            if response.status_code == 200:
                data = response.json()
                return {
                    'id': data.get('id'),
                    'symbol': data.get('symbol', '').upper(),
                    'name': data.get('name'),
                    'current_price': data.get('market_data', {}).get('current_price', {}),
                    'market_cap': data.get('market_data', {}).get('market_cap', {}),
                    'volume_24h': data.get('market_data', {}).get('total_volume', {}),
                    'price_change_24h': data.get('market_data', {}).get('price_change_percentage_24h'),
                    'last_updated': data.get('last_updated')
                }
            
            return None
            
        except Exception as e:
            print(f"Ошибка получения информации о токене: {e}")
            return None
    
    def clear_cache(self):
        """Очищает кэш цен"""
        self.price_cache.clear()
    
    def get_cache_stats(self) -> Dict[str, Any]:
        """Возвращает статистику кэша"""
        current_time = time.time()
        valid_entries = 0
        expired_entries = 0
        
        for cache_key, (price, timestamp) in self.price_cache.items():
            if current_time - timestamp < self.cache_ttl:
                valid_entries += 1
            else:
                expired_entries += 1
        
        return {
            'total_entries': len(self.price_cache),
            'valid_entries': valid_entries,
            'expired_entries': expired_entries,
            'cache_ttl': self.cache_ttl
        }
    
    def test_connection(self) -> Dict[str, bool]:
        """Тестирует подключение к API"""
        results = {}
        
        # Тест CoinGecko
        try:
            # Сбрасываем last_request для теста
            self.apis['coingecko']['last_request'] = 0
            test_price = self._get_coingecko_price('BTC', 'usd')
            results['coingecko'] = test_price is not None
        except:
            results['coingecko'] = False
        
        # Тест CoinMarketCap
        try:
            if self.apis['coinmarketcap']['api_key'] and self.apis['coinmarketcap']['api_key'] != 'YOUR_COINMARKETCAP_API_KEY':
                test_price = self._get_coinmarketcap_price('BTC', 'usd')
                results['coinmarketcap'] = test_price is not None
            else:
                results['coinmarketcap'] = False
        except:
            results['coinmarketcap'] = False
        
        return results
        
    def _get_fallback_price(self, token_symbol: str, currency: str = 'usd') -> Optional[float]:
        """Возвращает fallback цены для основных токенов"""
        # Fallback цены (примерные, обновляются редко)
        fallback_prices_usd = {
            'BTC': 110000.0,
            'ETH': 4000.0,
            'USDT': 1.0,
            'USDC': 1.0,
            'DAI': 1.0,
            'BNB': 800.0,
            'POL': 0.20,  # Актуальная цена Polygon (MATIC) - обновлено 2025-10-21
            'SOL': 100.0,
            'XRP': 0.5,
            'ADA': 0.4,
            'AVAX': 30.0,
            'DOGE': 0.08,
            'TRX': 0.1,
            'DOT': 6.0,
            'LINK': 15.0,
            'TON': 2.0,
            'SHIB': 0.00001,
            'LTC': 70.0,
            'BCH': 400.0,
            'ICP': 10.0,
            'UNI': 6.0,
            'NEAR': 5.0,
            'KAS': 0.1,
            'LEO': 4.0,
            'OKB': 50.0,
            'XLM': 0.1,
            'ETC': 25.0,
            'XMR': 150.0,
            'APT': 8.0,
            'HBAR': 0.05,
            'CRO': 0.1,
            'VET': 0.02,
            'FIL': 5.0,
            'ATOM': 8.0,
            'ARB': 1.5,
            'RNDR': 3.0,
            'IMX': 2.0,
            'GRT': 0.2,
            'OP': 2.5,
            'INJ': 30.0,
            'THETA': 1.5,
            'XTZ': 1.0,
            'SEI': 0.3,
            'FTM': 0.3,
            'LDO': 2.5,
            'TIA': 5.0,
            'BONK': 0.00002,
            'WBTC': 110000.0,
            'STX': 1.5,
            'SUI': 1.0,
            'EGLD': 30.0,
            'AXS': 6.0,
            'SAND': 0.4,
            'MANA': 0.4,
            'FLOW': 0.8,
            'APE': 1.5,
            'CHZ': 0.1,
            'MINA': 0.5,
            'GALA': 0.02,
            'MKR': 2000.0,
            'SNX': 3.0,
            'AAVE': 100.0,
            'COMP': 60.0,
            'YFI': 8000.0,
            'CRV': 0.5,
            'BAL': 4.0,
            '1INCH': 0.4,
            'ZRX': 0.3,
            'KAVA': 0.8,
            'SUSHI': 1.0,
            'RUNE': 5.0,
            'LRC': 0.2,
            'ENS': 15.0,
            'DYDX': 2.0,
            'GMX': 40.0,
            'CAKE': 2.0,
            'TWT': 1.0,
            'CVX': 3.0,
            'FXS': 6.0,
            'OCEAN': 0.5,
            'FET': 0.5,
            'AGIX': 0.3,
            'ROSE': 0.1,
            'CELO': 0.5,
            'SKL': 0.1,
            'RLC': 2.0,
            'BAND': 1.5,
            'ANKR': 0.03,
            'OMI': 0.001,
            'STORJ': 0.5,
            'UMA': 3.0,
            'POLY': 0.1,
            'COTI': 0.1,
            'POLS': 0.5,
            'GLM': 0.3,
            'PAXG': 2000.0,
            'USDP': 1.0,
            'GUSD': 1.0,
            'BUSD': 1.0
        }
        
        # Fallback цены в рублях (примерные)
        fallback_prices_rub = {
            'BTC': 10000000.0,
            'ETH': 360000.0,
            'USDT': 90.0,
            'USDC': 90.0,
            'BNB': 72000.0,
            'POL': 72.0,
            'SOL': 9000.0,
            'XRP': 45.0
        }
        
        token_upper = token_symbol.upper()
        
        if currency.lower() == 'usd':
            return fallback_prices_usd.get(token_upper)
        elif currency.lower() == 'rub':
            return fallback_prices_rub.get(token_upper)
        
        return None


class Plugin:
    """
    Плагин для сервиса цен токенов
    """
    def __init__(self, main_window):
        self.main_window = main_window
        self.dependencies = ["requests"]
    
    def get_instance(self):
        """Возвращает экземпляр сервиса цен токенов"""
        return TokenPriceService()
