import os
import json
import datetime
from typing import Optional, Dict, Any, List, Tuple

import requests
try:
    from requests_oauthlib import OAuth1
except ImportError:
    OAuth1 = None

from config.settings import settings_manager


class Plugin:
    """
    Плагин социальных сетей: публикация постов в X.com (Twitter) через Function Calling
    и быстрые команды. При необходимости можно расширить другими платформами.
    """

    def __init__(self, main_window=None):
        self.main_window = main_window
        self.dependencies = ["requests", "requests-oauthlib"]
        self.config_path = os.path.join("config", "social_media.txt")
        # Кэш последних публикаций (анти-дубликаты)
        self._dedupe_cache_path = os.path.join("config", "social_media_post_cache.json")
        self._dedupe_max_items = 20
        self._recent_post_texts: List[str] = []
        self._load_post_cache()
        self.session = requests.Session()
        # Устанавливаем стандартные заголовки для X API
        self.session.headers.update({
            'User-Agent': 'AIagent/1.0',
            'Accept': 'application/json'
        })
        self._last_verified_at: Optional[datetime.datetime] = None
        self._last_verified_user: Optional[str] = None
        self._verification_ttl_seconds = 600  # подтверждаем ключи раз в 10 минут
        self.required_credential_fields = [
            "api_key",
            "api_secret",
            "access_token",
            "access_token_secret",
        ]
        self.platform_aliases = {
            "twitter": "x",
            "x": "x",
            "x.com": "x",
            "твиттер": "x",
            "икс": "x"
        }
        self.tool_definitions = [
            {
                "type": "function",
                "function": {
                    "name": "social_post_message",
                    "description": "Публикация сообщений в социальных сетях (X.com). Используй после генерации текста поста.",
                    "parameters": {
                        "type": "object",
                        "properties": {
                            "platform": {
                                "type": "string",
                                "enum": ["x", "twitter", "x.com"],
                                "description": "Соцсеть для публикации (по умолчанию X.com)."
                            },
                            "text": {
                                "type": "string",
                                "description": "Готовый текст поста, который нужно отправить.",
                            },
                            "media_urls": {
                                "type": "array",
                                "items": {"type": "string"},
                                "description": "Список ссылок на изображения/видео (будут добавлены в конец текста)."
                            },
                            "tags": {
                                "type": "array",
                                "items": {"type": "string"},
                                "description": "Хэштеги без символа #. Будут добавлены в конец текста."
                            },
                            "scheduled_time": {
                                "type": "string",
                                "description": "Опционально: ISO-время для отложенной публикации. Пока только логируется."
                            },
                            "dry_run": {
                                "type": "boolean",
                                "description": "ТОЛЬКО для тестирования: если true — имитация без реальной отправки. По умолчанию НЕ передавай этот параметр - публикация будет реальной."
                            }
                        },
                        "required": ["text"],
                        "additionalProperties": False
                    }
                }
            }
        ]

    # === Основная интеграция =================================================

    def _load_post_cache(self) -> None:
        """Загружает кэш последних постов для анти-дубликатов."""
        try:
            if not os.path.exists(self._dedupe_cache_path):
                self._recent_post_texts = []
                return
            with open(self._dedupe_cache_path, "r", encoding="utf-8") as fh:
                data = json.load(fh) or {}
            items = data.get("recent_texts", [])
            if isinstance(items, list):
                self._recent_post_texts = [str(x) for x in items if str(x).strip()]
            else:
                self._recent_post_texts = []
            # ограничиваем размер
            self._recent_post_texts = self._recent_post_texts[-self._dedupe_max_items:]
        except Exception as exc:
            print(f"🔧 [social_media] ⚠️ Не удалось загрузить post cache: {exc}")
            self._recent_post_texts = []

    def _save_post_cache(self) -> None:
        """Сохраняет кэш последних постов на диск."""
        try:
            os.makedirs(os.path.dirname(self._dedupe_cache_path), exist_ok=True)
            payload = {
                "updated_at": datetime.datetime.utcnow().isoformat() + "Z",
                "recent_texts": self._recent_post_texts[-self._dedupe_max_items:],
            }
            with open(self._dedupe_cache_path, "w", encoding="utf-8") as fh:
                json.dump(payload, fh, ensure_ascii=False, indent=2)
        except Exception as exc:
            print(f"🔧 [social_media] ⚠️ Не удалось сохранить post cache: {exc}")

    def _is_duplicate_text(self, prepared_text: str) -> bool:
        """Проверка на дубликат по уже подготовленному тексту (с хэштегами/медиа)."""
        normalized = (prepared_text or "").strip()
        if not normalized:
            return False
        return normalized in self._recent_post_texts

    def _remember_post_text(self, prepared_text: str) -> None:
        normalized = (prepared_text or "").strip()
        if not normalized:
            return
        # удаляем возможный дубль и добавляем в конец
        self._recent_post_texts = [t for t in self._recent_post_texts if t != normalized]
        self._recent_post_texts.append(normalized)
        self._recent_post_texts = self._recent_post_texts[-self._dedupe_max_items:]
        self._save_post_cache()

    def handle_tool_call(self, function_name: str, arguments: Dict[str, Any]) -> Tuple[bool, str]:
        """Обработка Function Calling."""
        print(f"🔧 [social_media] handle_tool_call вызван: function_name={function_name}, arguments={arguments}")
        
        if function_name != "social_post_message":
            return False, f"Плагин соцсетей не поддерживает функцию '{function_name}'"

        text = (arguments.get("text") or "").strip()
        if not text:
            print(f"🔧 [social_media] ❌ Не указан текст поста")
            return False, "Не указан текст поста для публикации."

        platform = self._normalize_platform(arguments.get("platform", "x"))
        media_urls = self._normalize_list(arguments.get("media_urls"))
        tags = self._normalize_list(arguments.get("tags"))
        scheduled_time = arguments.get("scheduled_time")
        raw_dry_run = arguments.get("dry_run")
        # ВСЕГДА отключаем dry_run для Function Calling - это реальная публикация
        # LLM может присылать dry_run=true, но для проекта публикации должны быть реальными
        dry_run = False
        print(f"🔧 [social_media] handle_tool_call: dry_run принудительно отключен для Function Calling (было: {raw_dry_run})")
        print(f"🔧 [social_media] Параметры публикации: platform={platform}, text_length={len(text)}, tags={tags}")

        if platform == "x":
            print(f"🔧 [social_media] Вызываю _post_to_x для публикации в X.com")
            result = self._post_to_x(
                text,
                media_urls,
                tags,
                scheduled_time,
                dry_run,
                force_preview=False
            )
            print(f"🔧 [social_media] Результат _post_to_x: success={result[0]}, message={result[1][:100]}...")
            return result

        return False, f"Платформа '{platform}' пока не поддерживается."

    def execute_command(self, user_message: str) -> str:
        """
        Быстрые команды без Function Calling:
        примеры: "сделай пост в x.com: Текст...", "опубликуй твит ..."
        """
        if not user_message:
            return "❓ Нет текста для команды."

        message = user_message.strip()
        message_lower = message.lower()
        trigger_words = ["пост", "опубликуй", "запости", "tweet", "твит", "твиттер", "x.com", "x "]  # noqa: E501

        if not any(word in message_lower for word in trigger_words):
            return "❓ Команда соцсетей не распознана."

        platform = "x"
        text_part = message

        if ":" in message:
            text_part = message.split(":", 1)[1].strip()
        elif "—" in message:
            text_part = message.split("—", 1)[1].strip()

        if not text_part:
            return "❓ Уточните текст поста после двоеточия."

        success, response = self._post_to_x(
            text_part,
            media_urls=None,
            tags=None,
            scheduled_time=None,
            dry_run=True,  # Быстрые команды делаем безопасно
            force_preview=True
        )

        return response

    # === Публикация в X ======================================================

    def _post_to_x(
        self,
        text: str,
        media_urls: Optional[List[str]],
        tags: Optional[List[str]],
        scheduled_time: Optional[str],
        dry_run: bool,
        force_preview: bool = False
    ) -> Tuple[bool, str]:
        """Публикует сообщение в X.com."""
        if dry_run and not force_preview:
            print("🔧 [social_media] ⚠️ dry_run активирован, но публикации через Function Calling должны быть боевыми. Принудительно отключаю dry_run.")
            dry_run = False
        print(f"🔧 [social_media] _post_to_x вызван: dry_run={dry_run}, text_length={len(text)}, force_preview={force_preview}")
        creds = self._get_twitter_credentials()
        print(f"🔧 [social_media] Ключи получены: {bool(creds)}")
        if not creds:
            # Проверяем, какие ключи есть, чтобы дать более информативное сообщение
            settings_creds = settings_manager.get_social_credentials("x")
            config = self._load_social_config()
            missing = []
            invalid = []
            required_map = {
                "api_key": "TWITTER_API_KEY",
                "api_secret": "TWITTER_API_SECRET",
                "access_token": "TWITTER_ACCESS_TOKEN",
                "access_token_secret": "TWITTER_ACCESS_TOKEN_SECRET",
                "bearer_token": "TWITTER_BEARER_TOKEN",
            }
            for field, config_key in required_map.items():
                value_from_settings = settings_creds.get(field) if settings_creds else None
                value_from_config = config.get(config_key)
                value = value_from_settings or value_from_config
                
                if not value:
                    missing.append(field)
                elif len(str(value).strip()) < 10 or str(value).strip() in ["t:", "Bearer token:"]:
                    invalid.append(field)
            
            if invalid:
                return False, f"❌ Некорректные API ключи для X.com: {', '.join(invalid)}. Ключи слишком короткие или содержат некорректные значения. Проверьте ключи и добавьте правильные значения."
            if missing:
                return False, f"❌ Отсутствуют необходимые API ключи для X.com: {', '.join(missing)}. Добавьте их через чат или в config/social_media.txt"
            return False, "❌ Не найдены API ключи для X.com (ни в настройках, ни в config/social_media.txt). Нужны: api_key, api_secret, access_token, access_token_secret, bearer_token."

        prepared_text = self._prepare_text(text, tags, media_urls)
        # Анти-дубликаты ДО отправки в X: X может вернуть 403 duplicate content
        if self._is_duplicate_text(prepared_text):
            return False, "❌ Дубликат: этот текст уже публиковался. Измени содержимое (добавь уникальную деталь/фразу/число)."

        if scheduled_time:
            try:
                datetime.datetime.fromisoformat(scheduled_time.replace("Z", "+00:00"))
            except ValueError:
                return False, f"❌ Некорректное значение scheduled_time: {scheduled_time}"

        if dry_run:
            preview = {
                "platform": "x",
                "text": prepared_text,
                "scheduled_time": scheduled_time,
                "media_urls": media_urls or [],
                "tags": tags or []
            }
            return True, f"📝 DRY-RUN публикации в X:\n```json\n{json.dumps(preview, ensure_ascii=False, indent=2)}\n```"

        if OAuth1 is None:
            return False, "❌ Библиотека requests-oauthlib не установлена. Установите её и перезапустите агент."

        if not dry_run and self._needs_reverification():
            verified, verify_msg, username = self._verify_twitter_credentials(creds)
            if not verified:
                return False, verify_msg
            self._last_verified_at = datetime.datetime.utcnow()
            self._last_verified_user = username
            info_suffix = f" для @{username}" if username else ""
            print(f"🔧 [social_media] ✅ Ключи подтверждены{info_suffix} (ttl {self._verification_ttl_seconds}s)")

        # Проверяем наличие всех необходимых ключей
        required_keys = ["api_key", "api_secret", "access_token"]
        missing_keys = [key for key in required_keys if not creds.get(key)]
        if missing_keys:
            return False, f"❌ Отсутствуют необходимые ключи: {', '.join(missing_keys)}"
        
        # Используем access_secret или access_token_secret (для обратной совместимости)
        access_secret = creds.get("access_secret") or creds.get("access_token_secret")
        if not access_secret:
            return False, "❌ Отсутствует access_secret (или access_token_secret) для OAuth авторизации"

        try:
            # ВАЖНО: Для POST /2/tweets требуется OAuth 1.0a (user context)
            # Bearer Token используется только для app-only context (чтение данных, без публикации)
            # Поэтому используем только OAuth 1.0a для публикации постов
            
            # Формируем payload для API v2
            payload = {"text": prepared_text}
            
            # Используем OAuth 1.0a (требуется для публикации постов)
            auth = OAuth1(
                creds["api_key"],
                creds["api_secret"],
                creds["access_token"],
                access_secret
            )
            
            # ВАЖНО: Используем requests.post напрямую, как в тестовом скрипте
            # Это может помочь избежать проблем с сессией
            # НЕ добавляем заголовки вручную - OAuth1 сам добавит нужные заголовки
            # ВАЖНО: запоминаем текст как "уже отправлявшийся", чтобы не слать один и тот же пост повторно
            # (особенно при 403 duplicate content или при ретраях пользователя).
            self._remember_post_text(prepared_text)
            print(f"🔧 [social_media] Отправляю запрос: payload={json.dumps(payload, ensure_ascii=False)}, text_length={len(prepared_text)}")
            response = requests.post(
                "https://api.twitter.com/2/tweets",
                json=payload,
                auth=auth,
                timeout=20
            )
            # Всегда печатаем заголовки (часто там лимиты/причины отказа)
            print(f"🔧 [social_media] Ответ получен: status={response.status_code}, headers={dict(response.headers)}")
            print(f"🔧 [social_media] POST /2/tweets (OAuth 1.0a) -> {response.status_code}")
            # Дополнительная диагностика для отладки
            if response.status_code != 201 and response.status_code != 200:
                print(f"🔧 [social_media] ⚠️ НЕУСПЕШНЫЙ статус: {response.status_code}")
                try:
                    body_preview = (response.text or "")[:2000]
                except Exception:
                    body_preview = "<unavailable>"
                print(f"🔧 [social_media] Полный ответ: {body_preview}")
            
            if response.status_code in (200, 201):
                data = response.json()
                tweet_id = data.get("data", {}).get("id")
                url = f"https://x.com/i/web/status/{tweet_id}" if tweet_id else "https://x.com"
                print(f"🔧 [social_media] ✅ Публикация успешна через API v2, tweet_id={tweet_id}")
                return True, f"✅ Пост опубликован: {url}"
            
            # 429: rate limit / суточный лимит публикаций
            if response.status_code == 429:
                headers = dict(response.headers or {})
                # Заголовки X могут отличаться по продукту/плану:
                # - классические: x-rate-limit-limit/remaining/reset (+ retry-after)
                # - суточные: x-app-limit-24hour-*/x-user-limit-24hour-*
                app_rem = headers.get("x-app-limit-24hour-remaining")
                user_rem = headers.get("x-user-limit-24hour-remaining")
                app_reset = headers.get("x-app-limit-24hour-reset")
                user_reset = headers.get("x-user-limit-24hour-reset")
                rl_limit = headers.get("x-rate-limit-limit")
                rl_remaining = headers.get("x-rate-limit-remaining")
                rl_reset = headers.get("x-rate-limit-reset")
                retry_after = headers.get("retry-after")

                def _format_epoch(epoch_val: Any) -> str:
                    try:
                        if epoch_val is None:
                            return ""
                        epoch_int = int(str(epoch_val).strip())
                        # epoch seconds -> UTC datetime
                        dt = datetime.datetime.fromtimestamp(epoch_int, tz=datetime.timezone.utc)
                        return dt.isoformat()
                    except Exception:
                        return ""

                reset_app_iso = _format_epoch(app_reset)
                reset_user_iso = _format_epoch(user_reset)
                reset_rl_iso = _format_epoch(rl_reset)
                detail = "Too Many Requests"
                try:
                    j = response.json()
                    detail = j.get("detail") or j.get("title") or detail
                except Exception:
                    pass
                return False, (
                    "❌ Ошибка X API v2 (429): Too Many Requests.\n"
                    f"Детали: {detail}\n"
                    f"Retry-After: {retry_after}\n"
                    f"x-rate-limit (limit/remaining/reset): {rl_limit}/{rl_remaining}/{rl_reset} {reset_rl_iso}\n"
                    f"24h app/user remaining: {app_rem}/{user_rem}\n"
                    f"24h app/user reset (epoch -> UTC): {app_reset} {reset_app_iso} / {user_reset} {reset_user_iso}\n"
                    f"Ответ (preview): {(response.text or '')[:500]}"
                )

            # Если получили 403 от v2, проверяем детали ошибки
            if response.status_code == 403:
                print(f"🔧 [social_media] ⚠️ API v2 вернул 403, анализирую ошибку...")
                error_data = {}
                error_msg = ""
                error_code = ""
                error_detail = ""
                error_title = ""
                error_status = ""
                
                try:
                    error_data = response.json()
                    print(f"🔧 [social_media] Полный ответ ошибки: {json.dumps(error_data, ensure_ascii=False)}")
                    
                    # Обрабатываем два формата ошибок:
                    # 1. Старый формат: {"errors": [{"message": "...", "code": ...}]}
                    # 2. Новый формат: {"detail": "...", "title": "...", "status": 403}
                    errors = error_data.get('errors', [])
                    if errors:
                        error_msg = errors[0].get('message', '')
                        error_code = str(errors[0].get('code', ''))
                        print(f"🔧 [social_media] Формат errors: код={error_code}, сообщение={error_msg}")
                    else:
                        # Новый формат ошибки
                        error_detail = error_data.get('detail', '')
                        error_title = error_data.get('title', '')
                        error_status = error_data.get('status', '')
                        error_msg = error_detail or error_title
                        print(f"🔧 [social_media] Формат detail/title: title={error_title}, detail={error_detail}, status={error_status}")

                        # Дубликат контента — отдельная понятная ошибка (самая частая причина 403)
                        if "duplicate content" in (error_detail or "").lower():
                            return False, "❌ Ошибка: X API запретил дубликат текста (duplicate content). Измени текст или добавь уникальную деталь."
                        
                        # ВАЖНО: Если API v2 возвращает 403 с "You are not permitted" - это может быть код 453
                        # Проверяем различные признаки ошибки 453 (ограниченный доступ)
                        error_text_lower = (error_detail + " " + error_title).lower()
                        
                        # Признаки ошибки 453 (ограниченный доступ):
                        is_453_error = (
                            '453' in str(error_status) or 
                            '453' in error_detail or
                            'subset of x api v2 endpoints' in error_text_lower or
                            'limited access' in error_text_lower or
                            ('application has limited access' in error_text_lower and error_status == 403) or
                            ('restricted access' in error_text_lower and error_status == 403)
                        )
                        
                        if is_453_error:
                            error_code = '453'
                            print(f"🔧 [social_media] Обнаружен код 453 (ограниченный доступ) в новом формате ошибки")
                        elif 'not permitted' in error_detail.lower() and error_status == 403:
                            # Если v2 возвращает 403 с "not permitted" - это может быть код 453
                            # Проверяем, есть ли признаки ограниченного доступа в тексте
                            if 'limited' in error_text_lower or 'restricted' in error_text_lower:
                                error_code = '453'
                                print(f"🔧 [social_media] Обнаружен код 453 по признакам 'limited/restricted' в тексте ошибки")
                            else:
                                # НЕ устанавливаем error_code = '453' сразу, чтобы дать шанс v1.1
                                print(f"🔧 [social_media] Обнаружена ошибка 'not permitted' на v2 (403) - попробую v1.1")
                        
                        # Если есть "You are not permitted" - это проблема с правами
                        if 'not permitted' in error_detail.lower() or 'forbidden' in error_title.lower():
                            error_msg = error_detail or error_title
                            print(f"🔧 [social_media] Обнаружена ошибка прав доступа: {error_msg}")
                except Exception as e:
                    print(f"🔧 [social_media] Ошибка парсинга JSON: {e}")
                    error_preview = response.text[:500]
                    print(f"🔧 [social_media] Сырой ответ: {error_preview}")
                error_preview = response.text[:500]
                
                # Формируем понятное сообщение об ошибке
                if error_code == '453' or 'subset of X API V2 endpoints' in error_msg:
                    return False, f"❌ Ошибка X API v2 (403, код {error_code}): Приложение имеет ограниченный доступ к API.\n\n💡 Ваше приложение имеет доступ только к подмножеству X API V2 endpoints.\n\n📋 Решения:\n1. Проверьте тип приложения в X Developer Portal:\n   - Зайдите в https://developer.x.com/en/portal/dashboard\n   - Выберите ваше приложение → Settings\n   - Убедитесь, что тип приложения \"Web App / Bot\" (не \"Read-only\")\n\n2. Проверьте права доступа:\n   - Settings → User authentication settings\n   - Должно быть выбрано \"Read and write\"\n   - Сохраните изменения\n\n3. Пересоздайте токены после изменения прав:\n   - Keys and tokens → Regenerate Access Token и Access Token Secret\n   - Обновите ключи в AIagent\n\n4. Если проблема сохраняется, возможно требуется обновление плана API:\n   - https://developer.x.com/en/portal/products\n\n📖 Подробнее: https://developer.x.com/en/portal/product"
                elif 'not permitted' in error_msg.lower() or 'forbidden' in error_msg.lower():
                    return False, f"❌ Ошибка X API v2 (403): {error_msg or error_title or 'You are not permitted to perform this action'}\n\n💡 Проблема с правами доступа приложения.\n\n📋 Решения:\n1. Проверьте права приложения в X Developer Portal:\n   - Зайдите в https://developer.x.com/en/portal/dashboard\n   - Выберите ваше приложение → Settings → User authentication settings\n   - Убедитесь, что выбрано \"Read and write\" (не только \"Read\")\n   - Сохраните изменения\n\n2. Пересоздайте токены после изменения прав:\n   - В настройках приложения найдите \"Keys and tokens\"\n   - Удалите старые Access Token и Access Token Secret\n   - Создайте новые токены с правами \"Read and write\"\n   - Обновите ключи в AIagent\n\n3. Проверьте тип приложения:\n   - Должен быть тип \"Web App / Bot\" (не \"Read-only\")\n\n📖 Подробнее: https://developer.x.com/en/portal/products"
                else:
                    return False, f"❌ Ошибка X API v2 (403): {error_msg or error_detail or error_title or error_preview}\n\n💡 Проверьте права приложения в X Developer Portal."
            
            error_preview = response.text[:500]
            return False, f"❌ Ошибка X API v2 ({response.status_code}): {error_preview}"
        except requests.exceptions.RequestException as exc:
            return False, f"❌ Сетевая ошибка отправки поста: {exc}"

    # === Вспомогательные методы =============================================

    def _prepare_text(self, text: str, tags: Optional[List[str]], media_urls: Optional[List[str]]) -> str:
        prepared = text.strip()

        if tags:
            cleaned_tags = [tag if tag.startswith("#") else f"#{tag}" for tag in tags if tag]
            if cleaned_tags:
                prepared += "\n\n" + " ".join(cleaned_tags)

        if media_urls:
            block = "\n".join(media_urls)
            prepared += f"\n\n{block}"

        return prepared

    def _normalize_list(self, value: Any) -> Optional[List[str]]:
        if value is None:
            return None
        if isinstance(value, list):
            return [str(item).strip() for item in value if str(item).strip()]
        if isinstance(value, str):
            return [item.strip() for item in value.split(",") if item.strip()]
        return None

    def _normalize_platform(self, platform: Optional[str]) -> str:
        if not platform:
            return "x"
        key = platform.strip().lower()
        return self.platform_aliases.get(key, key)

    def _get_twitter_credentials(self) -> Optional[Dict[str, str]]:
        print(f"🔧 [social_media] Загружаю ключи X.com из настроек...")
        settings_creds = settings_manager.get_social_credentials("x")
        if settings_creds:
            print(f"🔧 [social_media] Ключи найдены в настройках: {list(settings_creds.keys())}")
            normalized = {
                "api_key": settings_creds.get("api_key"),
                "api_secret": settings_creds.get("api_secret"),
                "access_token": settings_creds.get("access_token"),
                "access_token_secret": settings_creds.get("access_token_secret"),
                "access_secret": settings_creds.get("access_token_secret"),
                "bearer_token": settings_creds.get("bearer_token"),
            }
            # Логируем первые и последние символы ключей для проверки (безопасно)
            if normalized.get("api_key"):
                api_key = normalized["api_key"]
                print(f"🔧 [social_media] API Key: {api_key[:5]}...{api_key[-5:] if len(api_key) > 10 else '***'}")
            if normalized.get("access_token"):
                access_token = normalized["access_token"]
                print(f"🔧 [social_media] Access Token: {access_token[:10]}...{access_token[-5:] if len(access_token) > 15 else '***'}")
            
            valid, missing, invalid = self._validate_required_credentials(normalized)
            if valid:
                if not normalized.get("bearer_token"):
                    print("🔧 [social_media] ⚠️ Bearer token не найден — некоторые запросы X API могут быть недоступны")
                print("🔧 [social_media] ✅ Найдены валидные ключи в настройках")
                # Дополнительная проверка: все ключи должны быть непустыми
                all_keys_present = all([
                    normalized.get("api_key"),
                    normalized.get("api_secret"),
                    normalized.get("access_token"),
                    normalized.get("access_token_secret")
                ])
                if not all_keys_present:
                    print(f"🔧 [social_media] ⚠️ ВНИМАНИЕ: Не все ключи присутствуют! api_key={bool(normalized.get('api_key'))}, api_secret={bool(normalized.get('api_secret'))}, access_token={bool(normalized.get('access_token'))}, access_token_secret={bool(normalized.get('access_token_secret'))}")
                return normalized
            print(f"🔧 [social_media] ⚠️ Ключи в настройках некорректны (missing={missing}, invalid={invalid}), пробую config файл")

        config = self._load_social_config()
        keys = {
            "api_key": config.get("TWITTER_API_KEY"),
            "api_secret": config.get("TWITTER_API_SECRET"),
            "access_token": config.get("TWITTER_ACCESS_TOKEN"),
            "access_secret": config.get("TWITTER_ACCESS_TOKEN_SECRET"),  # Используем access_secret для консистентности
            "access_token_secret": config.get("TWITTER_ACCESS_TOKEN_SECRET"),  # Оставляем для обратной совместимости
            "bearer_token": config.get("TWITTER_BEARER_TOKEN")
        }
        valid, missing, invalid = self._validate_required_credentials(keys)
        if valid:
            if not keys.get("bearer_token"):
                print("🔧 [social_media] ⚠️ Bearer token отсутствует в config — проверьте X API доступы")
            print("🔧 [social_media] ✅ Найдены валидные ключи в config файле")
            return keys

        print(f"🔧 [social_media] ❌ Ключи не найдены или некорректны ни в настройках, ни в config (missing={missing}, invalid={invalid})")
        return None

    def _validate_required_credentials(self, creds: Dict[str, Optional[str]]) -> Tuple[bool, List[str], List[str]]:
        missing: List[str] = []
        invalid: List[str] = []
        for field in self.required_credential_fields:
            value = creds.get(field) or creds.get(field.replace("_secret", "_token_secret"))
            if not value:
                missing.append(field)
            elif len(str(value).strip()) < 10 or str(value).strip() in ["t:", "Bearer token:"]:
                invalid.append(field)
        return not missing and not invalid, missing, invalid

    def _needs_reverification(self) -> bool:
        if not self._last_verified_at:
            return True
        delta = datetime.datetime.utcnow() - self._last_verified_at
        return delta.total_seconds() > self._verification_ttl_seconds

    def _verify_twitter_credentials(self, creds: Dict[str, str]) -> Tuple[bool, str, Optional[str]]:
        """
        Лёгкий self-check, рекомендуемый в гайде X API: убеждаемся, что ключи рабочие,
        прежде чем публиковать пост. Используем OAuth1 и endpoint account/verify_credentials.
        """
        if OAuth1 is None:
            return False, "❌ Невозможно подтвердить ключи: не установлена requests-oauthlib", None

        access_secret = creds.get("access_secret") or creds.get("access_token_secret")
        if not access_secret:
            return False, "❌ Не удаётся подтвердить ключи: отсутствует access_secret/access_token_secret", None

        try:
            auth = OAuth1(
                creds["api_key"],
                creds["api_secret"],
                creds["access_token"],
                access_secret
            )
            response = self.session.get(
                "https://api.twitter.com/1.1/account/verify_credentials.json",
                auth=auth,
                timeout=15
            )
            if response.status_code == 200:
                data = response.json()
                username = data.get("screen_name") or data.get("name")
                return True, "✅ Ключи подтверждены", username
            return False, f"❌ Не удалось подтвердить ключи ({response.status_code}): {response.text[:400]}", None
        except requests.exceptions.RequestException as exc:
            return False, f"❌ Сетевая ошибка при проверке ключей: {exc}", None

    def _load_social_config(self) -> Dict[str, str]:
        data: Dict[str, str] = {}
        if not os.path.exists(self.config_path):
            return data

        with open(self.config_path, "r", encoding="utf-8") as fh:
            for raw_line in fh:
                line = raw_line.strip()
                if not line or line.startswith("#") or "=" not in line:
                    continue
                key, value = line.split("=", 1)
                data[key.strip()] = value.strip()
        return data

