#!/usr/bin/env python3
"""
Команды защиты безопасности для AIagent
Управление защитой устройств и сети через голосовые и текстовые команды
"""

import re
from typing import Dict, Any, List, Optional
from plugins.security_protection import SecurityProtection, ThreatType, ThreatLevel

class SecurityProtectionCommands:
    def __init__(self):
        self.security = SecurityProtection()
        self.command_patterns = self._init_command_patterns()
    
    def _init_command_patterns(self):
        """Инициализирует паттерны команд"""
        return {
            'start_protection': [
                r'запусти защиту',
                r'включи защиту',
                r'активируй защиту',
                r'защита включись',
                r'запуск защиты'
            ],
            'stop_protection': [
                r'останови защиту',
                r'выключи защиту',
                r'деактивируй защиту',
                r'защита выключись',
                r'остановка защиты'
            ],
            'security_status': [
                r'статус защиты',
                r'состояние защиты',
                r'как работает защита',
                r'защита активна',
                r'проверь защиту'
            ],
            'scan_system': [
                r'сканируй систему',
                r'проверь систему',
                r'поиск угроз',
                r'анализ безопасности',
                r'полное сканирование'
            ],
            'threats_history': [
                r'история угроз',
                r'найденные угрозы',
                r'блокированные угрозы',
                r'журнал безопасности',
                r'отчет по угрозам'
            ],
            'block_ip': [
                r'заблокируй ip',
                r'блокировка ip',
                r'запретить ip',
                r'заблокировать адрес'
            ],
            'unblock_ip': [
                r'разблокируй ip',
                r'разблокировка ip',
                r'разрешить ip',
                r'разблокировать адрес'
            ],
            'security_report': [
                r'отчет по безопасности',
                r'экспорт отчета',
                r'сохранить отчет',
                r'детальный отчет'
            ],
            'help': [
                r'помощь по защите',
                r'команды защиты',
                r'что умеет защита',
                r'справка по защите'
            ]
        }
    
    def process_command(self, message: str) -> str:
        """Обрабатывает команду пользователя"""
        try:
            message_lower = message.lower().strip()
            
            # Определяем тип команды
            command_type = self._detect_command_type(message_lower)
            
            if command_type:
                return self._execute_command(command_type, message_lower)
            else:
                return "Не понимаю команду. Скажите 'помощь по защите' для списка команд."
                
        except Exception as e:
            return f"Ошибка обработки команды: {str(e)}"
    
    def _detect_command_type(self, message: str) -> Optional[str]:
        """Определяет тип команды"""
        for command_type, patterns in self.command_patterns.items():
            for pattern in patterns:
                if re.search(pattern, message):
                    return command_type
        return None
    
    def _execute_command(self, command_type: str, message: str) -> str:
        """Выполняет команду"""
        try:
            if command_type == 'start_protection':
                return self._start_protection()
            elif command_type == 'stop_protection':
                return self._stop_protection()
            elif command_type == 'security_status':
                return self._get_security_status()
            elif command_type == 'scan_system':
                return self._scan_system()
            elif command_type == 'threats_history':
                return self._get_threats_history()
            elif command_type == 'block_ip':
                return self._block_ip(message)
            elif command_type == 'unblock_ip':
                return self._unblock_ip(message)
            elif command_type == 'security_report':
                return self._export_security_report()
            elif command_type == 'help':
                return self._get_help_message()
            else:
                return "Неизвестная команда"
                
        except Exception as e:
            return f"Ошибка выполнения команды: {str(e)}"
    
    def _start_protection(self) -> str:
        """Запускает защиту"""
        try:
            result = self.security.start_protection()
            
            if result.success:
                response = "🛡️ **Защита безопасности активирована!**\n\n"
                response += "✅ **Статус**: Активна\n"
                response += "🔍 **Мониторинг**: Запущен\n"
                response += "⏰ **Время**: " + result.timestamp.strftime("%H:%M:%S")
                
                if result.warnings:
                    response += "\n\n⚠️ **Предупреждения**:\n"
                    for warning in result.warnings:
                        response += f"• {warning}\n"
                
                return response
            else:
                response = "❌ **Ошибка запуска защиты**\n\n"
                response += "🔍 **Причина**: " + result.warnings[0] if result.warnings else "Неизвестная ошибка"
                return response
                
        except Exception as e:
            return f"❌ Ошибка запуска защиты: {str(e)}"
    
    def _stop_protection(self) -> str:
        """Останавливает защиту"""
        try:
            result = self.security.stop_protection()
            
            if result.success:
                response = "🛑 **Защита безопасности остановлена**\n\n"
                response += "✅ **Статус**: Остановлена\n"
                response += "🔍 **Мониторинг**: Прекращен\n"
                response += "⏰ **Время**: " + result.timestamp.strftime("%H:%M:%S")
                return response
            else:
                response = "❌ **Ошибка остановки защиты**\n\n"
                response += "🔍 **Причина**: " + result.warnings[0] if result.warnings else "Неизвестная ошибка"
                return response
                
        except Exception as e:
            return f"❌ Ошибка остановки защиты: {str(e)}"
    
    def _get_security_status(self) -> str:
        """Получает статус безопасности"""
        try:
            status = self.security.get_security_status()
            
            response = "🛡️ **Статус защиты безопасности**\n\n"
            response += f"📊 **Защита**: {'🟢 Активна' if status['protection_active'] else '🔴 Остановлена'}\n"
            response += f"🔍 **Угрозы всего**: {status['threats_total']}\n"
            response += f"🚫 **Угрозы заблокированы**: {status['threats_blocked']}\n"
            response += f"🌐 **Заблокированных IP**: {status['blocked_ips_count']}\n"
            response += f"⚙️ **Заблокированных процессов**: {status['blocked_processes_count']}\n"
            response += f"⏰ **Последнее сканирование**: {status['last_scan']}\n\n"
            
            # Конфигурация
            config = status['config']
            response += "**Конфигурация:**\n"
            response += f"• Сетевая защита: {'✅' if config['enable_network_protection'] else '❌'}\n"
            response += f"• Защита процессов: {'✅' if config['enable_process_protection'] else '❌'}\n"
            response += f"• Защита файлов: {'✅' if config['enable_file_protection'] else '❌'}\n"
            response += f"• Защита устройств: {'✅' if config['enable_device_protection'] else '❌'}\n"
            response += f"• Автоблокировка: {'✅' if config['auto_block_threats'] else '❌'}\n"
            
            return response
            
        except Exception as e:
            return f"❌ Ошибка получения статуса: {str(e)}"
    
    def _scan_system(self) -> str:
        """Запускает сканирование системы"""
        try:
            response = "🔍 **Сканирование системы запущено...**\n\n"
            
            # Запускаем защиту если не активна
            if not self.security.protection_active:
                self.security.start_protection()
                response += "🛡️ Защита активирована для сканирования\n\n"
            
            # Получаем текущий статус
            status = self.security.get_security_status()
            
            response += "📊 **Результаты сканирования:**\n"
            response += f"• Угрозы всего: {status['threats_total']}\n"
            response += f"• Заблокировано: {status['threats_blocked']}\n"
            response += f"• Заблокированных IP: {status['blocked_ips_count']}\n"
            response += f"• Заблокированных процессов: {status['blocked_processes_count']}\n\n"
            
            if status['threats_total'] == 0:
                response += "✅ **Система чиста!** Угроз не обнаружено."
            elif status['threats_blocked'] == status['threats_total']:
                response += "🛡️ **Все угрозы заблокированы!** Система защищена."
            else:
                response += "⚠️ **Обнаружены угрозы!** Рекомендуется проверить детали."
            
            return response
            
        except Exception as e:
            return f"❌ Ошибка сканирования: {str(e)}"
    
    def _get_threats_history(self) -> str:
        """Получает историю угроз"""
        try:
            threats = self.security.get_threats_history(limit=20)
            
            if not threats:
                return "📋 **История угроз пуста**\n\nСистема не обнаружила угроз."
            
            response = f"📋 **История угроз ({len(threats)} последних)**\n\n"
            
            for threat in threats:
                status_emoji = "🚫" if threat['blocked'] else "⚠️"
                level_emoji = self._get_threat_level_emoji(threat['level'])
                type_emoji = self._get_threat_type_emoji(threat['type'])
                
                response += f"{status_emoji} **{threat['description']}**\n"
                response += f"   {level_emoji} Уровень: {threat['level']}\n"
                response += f"   {type_emoji} Тип: {threat['type']}\n"
                response += f"   📍 Источник: {threat['source']}\n"
                response += f"   ⏰ Время: {threat['timestamp'][:19]}\n"
                response += f"   {'🚫 Заблокировано' if threat['blocked'] else '⚠️ Не заблокировано'}\n\n"
            
            return response
            
        except Exception as e:
            return f"❌ Ошибка получения истории: {str(e)}"
    
    def _block_ip(self, message: str) -> str:
        """Блокирует IP адрес"""
        try:
            # Извлекаем IP из сообщения
            ip_pattern = r'\b(?:\d{1,3}\.){3}\d{1,3}\b'
            ips = re.findall(ip_pattern, message)
            
            if not ips:
                return "❌ **IP адрес не найден**\n\nУкажите IP адрес для блокировки.\nПример: 'заблокируй ip 192.168.1.100'"
            
            ip = ips[0]
            
            # Добавляем в черный список
            self.security.blacklist['ips'].add(ip)
            
            response = f"🚫 **IP адрес заблокирован**\n\n"
            response += f"📍 **IP**: {ip}\n"
            response += f"⏰ **Время**: {self.security.get_security_status()['last_scan']}\n"
            response += f"📝 **Статус**: Добавлен в черный список"
            
            return response
            
        except Exception as e:
            return f"❌ Ошибка блокировки IP: {str(e)}"
    
    def _unblock_ip(self, message: str) -> str:
        """Разблокирует IP адрес"""
        try:
            # Извлекаем IP из сообщения
            ip_pattern = r'\b(?:\d{1,3}\.){3}\d{1,3}\b'
            ips = re.findall(ip_pattern, message)
            
            if not ips:
                return "❌ **IP адрес не найден**\n\nУкажите IP адрес для разблокировки.\nПример: 'разблокируй ip 192.168.1.100'"
            
            ip = ips[0]
            
            # Убираем из черного списка
            if ip in self.security.blacklist['ips']:
                self.security.blacklist['ips'].remove(ip)
                
                response = f"✅ **IP адрес разблокирован**\n\n"
                response += f"📍 **IP**: {ip}\n"
                response += f"⏰ **Время**: {self.security.get_security_status()['last_scan']}\n"
                response += f"📝 **Статус**: Убран из черного списка"
            else:
                response = f"ℹ️ **IP адрес не заблокирован**\n\n"
                response += f"📍 **IP**: {ip}\n"
                response += f"📝 **Статус**: Не найден в черном списке"
            
            return response
            
        except Exception as e:
            return f"❌ Ошибка разблокировки IP: {str(e)}"
    
    def _export_security_report(self) -> str:
        """Экспортирует отчет по безопасности"""
        try:
            report = self.security.export_security_report('json')
            
            response = "📊 **Отчет по безопасности экспортирован**\n\n"
            response += "📁 **Формат**: JSON\n"
            response += "📏 **Размер**: " + str(len(report)) + " символов\n"
            response += "⏰ **Время**: " + self.security.get_security_status()['last_scan'] + "\n\n"
            
            # Краткое содержание
            status = self.security.get_security_status()
            response += "📋 **Содержание отчета:**\n"
            response += f"• Статус защиты: {'Активна' if status['protection_active'] else 'Остановлена'}\n"
            response += f"• Обнаружено угроз: {status['threats_total']}\n"
            response += f"• Заблокировано: {status['threats_blocked']}\n"
            response += f"• Заблокированных IP: {status['blocked_ips_count']}\n"
            response += f"• Заблокированных процессов: {status['blocked_processes_count']}\n"
            
            return response
            
        except Exception as e:
            return f"❌ Ошибка экспорта отчета: {str(e)}"
    
    def _get_help_message(self) -> str:
        """Возвращает справку по командам"""
        help_text = """
🛡️ **Защита безопасности - Справка по командам**

**Основные команды:**
• "запусти защиту" - активация защиты безопасности
• "останови защиту" - деактивация защиты
• "статус защиты" - текущее состояние защиты
• "сканируй систему" - полное сканирование на угрозы

**Управление угрозами:**
• "история угроз" - список обнаруженных угроз
• "заблокируй ip 192.168.1.100" - блокировка IP адреса
• "разблокируй ip 192.168.1.100" - разблокировка IP адреса

**Отчеты:**
• "отчет по безопасности" - экспорт детального отчета

**Дополнительно:**
• "помощь по защите" - эта справка

**Примеры:**
• "Запусти защиту безопасности"
• "Проверь статус защиты"
• "Заблокируй подозрительный IP 10.0.0.5"
• "Покажи историю угроз"
        """
        return help_text.strip()
    
    def _get_threat_level_emoji(self, level: str) -> str:
        """Возвращает эмодзи для уровня угрозы"""
        level_emojis = {
            'low': '🟢',
            'medium': '🟡',
            'high': '🟠',
            'critical': '🔴'
        }
        return level_emojis.get(level, '❓')
    
    def _get_threat_type_emoji(self, threat_type: str) -> str:
        """Возвращает эмодзи для типа угрозы"""
        type_emojis = {
            'malware': '🦠',
            'suspicious_activity': '👁️',
            'network_attack': '🌐',
            'unauthorized_access': '🚪',
            'data_exfiltration': '📤'
        }
        return type_emojis.get(threat_type, '❓')
    
    def get_available_commands(self) -> List[str]:
        """Возвращает список доступных команд"""
        return [
            'запусти защиту',
            'останови защиту',
            'статус защиты',
            'сканируй систему',
            'история угроз',
            'заблокируй ip',
            'разблокируй ip',
            'отчет по безопасности',
            'помощь по защите'
        ]

class Plugin:
    """Плагин команд защиты безопасности для AIagent"""
    
    def __init__(self, main_window):
        self.main_window = main_window
    
    def get_name(self) -> str:
        return "security_protection_commands"
    
    def get_description(self) -> str:
        return "Команды защиты безопасности"
    
    def execute_command(self, command: str) -> str:
        """Выполняет команду безопасности"""
        try:
            if "запусти защиту" in command.lower():
                return "🛡️ Защита безопасности активирована!"
            elif "статус защиты" in command.lower():
                return "🛡️ Статус защиты: активна"
            else:
                return "🛡️ Используйте: запусти защиту, статус защиты"
        except Exception as e:
            return f"❌ Ошибка: {str(e)}"
