import requests
from typing import Dict, Optional, Any
from web3 import Web3

class OneInchIntegration:
    def __init__(self):
        self.base_urls = {
            1: "https://api.1inch.io/v5.0/1",      # Ethereum
            56: "https://api.1inch.io/v5.0/56",    # BSC
            137: "https://api.1inch.io/v5.0/137",  # Polygon
            10: "https://api.1inch.io/v5.0/10",    # Optimism
            42161: "https://api.1inch.io/v5.0/42161"  # Arbitrum
        }
        
        # Адреса нативных токенов для 1inch
        self.native_tokens = {
            1: "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",  # ETH
            56: "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",  # BNB
            137: "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",  # POL
            10: "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee",  # ETH
            42161: "0xeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee"  # ETH
        }

    def get_quote(self, from_token: Dict, to_token: Dict, amount: float, chain_id: int) -> Optional[Dict]:
        """Получает котировку от 1inch API"""
        try:
            base_url = self.base_urls.get(chain_id)
            if not base_url:
                print(f"❌ 1inch не поддерживает сеть с ID {chain_id}")
                return None
            
            # Определяем адреса токенов
            from_address = self.native_tokens.get(chain_id) if from_token['address'] == '0x0000000000000000000000000000000000000000' else from_token['address']
            to_address = self.native_tokens.get(chain_id) if to_token['address'] == '0x0000000000000000000000000000000000000000' else to_token['address']
            
            # Конвертируем сумму в wei
            amount_wei = int(amount * (10**from_token['decimals']))
            
            url = f"{base_url}/quote"
            params = {
                'fromTokenAddress': from_address,
                'toTokenAddress': to_address,
                'amount': str(amount_wei)
            }
            
            response = requests.get(url, params=params, timeout=15)
            response.raise_for_status()
            data = response.json()
            
            # Вычисляем курс и количество
            amount_out = int(data['toTokenAmount']) / (10**to_token['decimals'])
            rate = amount_out / amount
            
            return {
                "rate": rate,
                "amount_out": amount_out,
                "tx_data": data,
                "dex": "1inch"
            }
            
        except requests.RequestException as e:
            print(f"❌ 1inch API ошибка: {e}")
            return None
        except Exception as e:
            print(f"❌ Ошибка получения котировки 1inch: {e}")
            return None

    def execute_swap(self, private_key: str, tx_data: Dict, rpc_url: str, chain_id: int) -> Dict:
        """Выполняет обмен через 1inch"""
        try:
            web3 = Web3(Web3.HTTPProvider(rpc_url))
            
            # Добавляем POA middleware для Polygon
            if chain_id == 137:
                try:
                    from web3.middleware import ExtraDataToPOAMiddleware
                    web3.middleware_onion.inject(ExtraDataToPOAMiddleware, layer=0)
                except ImportError:
                    try:
                        from web3.middleware import geth_poa_middleware
                        web3.middleware_onion.inject(geth_poa_middleware, layer=0)
                    except ImportError:
                        pass
            
            account = web3.eth.account.from_key(private_key)
            
            # Получаем параметры транзакции
            tx_params = {
                'from': account.address,
                'to': tx_data['to'],
                'data': tx_data['data'],
                'value': int(tx_data['value']),
                'gas': int(tx_data['gas']),
                'gasPrice': int(tx_data['gasPrice']),
                'nonce': web3.eth.get_transaction_count(account.address)
            }
            
            # Подписываем и отправляем транзакцию
            signed_tx = web3.eth.account.sign_transaction(tx_params, private_key)
            tx_hash = web3.eth.send_raw_transaction(signed_tx.rawTransaction)
            
            return {"success": True, "tx_hash": web3.to_hex(tx_hash)}
            
        except Exception as e:
            print(f"❌ Ошибка выполнения свопа 1inch: {e}")
            return {"success": False, "error": str(e)}

class Plugin:
    """Плагин интеграции с 1inch для AIagent"""
    
    def __init__(self, main_window):
        self.main_window = main_window
        self.oneinch = OneInchIntegration()
    
    def get_plugin_name(self):
        return "1inch Integration"
    
    def get_plugin_description(self):
        return "Интеграция с 1inch DEX агрегатором для обмена токенов"
