import json
from typing import Dict, Optional
from web3 import Web3
from config.settings import settings_manager
from concurrent.futures import ThreadPoolExecutor, as_completed

# Глобальная переменная для сигнала превышения лимита API (совместимость)
api_rate_limit_signal = None

def set_api_rate_limit_signal(signal):
    """Устанавливает сигнал для уведомления о превышении лимита API (совместимость с main.py)"""
    global api_rate_limit_signal
    api_rate_limit_signal = signal

class BalanceManager:
    """Управляет получением и форматированием балансов кошелька."""
    def __init__(self, wallet, settings_container):
        self.wallet = wallet
        self.settings_container = settings_container
        # Упрощенный ABI для получения баланса ERC20
        self.erc20_abi = json.loads('[{"constant":true,"inputs":[{"name":"_owner","type":"address"}],"name":"balanceOf","outputs":[{"name":"balance","type":"uint256"}],"payable":false,"stateMutability":"view","type":"function"},{"constant":true,"inputs":[],"name":"decimals","outputs":[{"name":"","type":"uint8"}],"payable":false,"stateMutability":"view","type":"function"}]')
        
        # Список топ-50 токенов для проверки (адреса для Polygon)
        # ⚠️ POL и MATIC удалены - это нативные токены, они получаются отдельно!
        self.token_contracts = {
            "USDT": "0xc2132D05D31c914a87C6611C10748AEb04B58e8F",
            "USDC": "0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174",
            "WBTC": "0x1BFD67037B42Cf73acF2047067bd4F2C47D9BfD6",
            "DAI": "0x8f3Cf7ad23Cd3CaDbD9735AFf958023239c6A063",
            "WETH": "0x7ceB23fD6bC0adD59E62ac25578270cFf1b9f619",
            "LINK": "0x53E0bca35eC356BD5ddDFebbD1Fc0fD03FaBad39",
            "AAVE": "0xD6DF932A45C0f255f85145f286eA0b292B21C90B",
            "UNI": "0xb33EaAd8d922B1083446DC23f610c2567fB5180f",
            "CRV": "0x172370d5Cd63279eFa6d502DAB29171933a610AF",
            # COMP и MKR удалены - не существуют на Polygon или адреса неверные
            "1INCH": "0x111111111117dC0aa78b770fA6A738034120C302",
            "SUSHI": "0x0b3F868E0BE5597D5DB7fEB59E1CADBb0fdDa50a",
            "BAL": "0x9a71012B13CA4d3D0CdC72A177DF3ef03b0E76A3",
            "YFI": "0xDA537104D6A5edd53c6fBba9A898708E465260b6",
            "SNX": "0x50B728D8D964fd00C2d0AAD81718b71311feF68a",
            "MATICX": "0xfa68FB4628DFF1028CFEc22b4162FCcd0d45efb6",
            "GHST": "0x385Eeac5cB85A38A9a07A70c73e0a3271CfB54A7",
            "WMATIC": "0x0d500B1d8E8eF31E21C99d1Db9A6444d3ADf1270",
            "QUICK": "0x831753DD7087CaC61aB5644b308642cc1c33Dc13",
            "META": "0x9C78EE466D6Cb57A4d01Fd887D2b5dFb2D46288f",
            "DPI": "0x85955046DF4668e1DD369D2DE9f3AEB98DD2A369",
            "PAXG": "0x553d3D295e0f695B9228246232eDF400ed3560B5",
            "USDP": "0x2e1AD108fF1D8C782fCbBb6e3568d81C85A31E0A",
            "WBTC.e": "0x1BFD67037B42Cf73acF2047067bd4F2C47D9BfD6",
            "EURS": "0xE111178A87A3BFf0c8d18DECBa5798827539Ae99",
        }
        self.web3 = None
        # Цены токенов загружаются асинхронно в UI, не здесь
        
        # Поддерживаемые сети 0x API v2 (все 12 сетей)
        self.networks = {
            "ethereum": {
                "name": "Ethereum",
                "chain_id": 1,
                "rpc_urls": [
                    "https://eth.llamarpc.com",
                    "https://rpc.ankr.com/eth",
                    "https://ethereum.publicnode.com"
                ],
                "native_token": "ETH",
                "explorer": "https://etherscan.io",
                "etherscan_chain_id": 1  # Для Etherscan API
            },
            "optimism": {
                "name": "Optimism",
                "chain_id": 10,
                "rpc_urls": [
                    "https://optimism.llamarpc.com",
                    "https://rpc.ankr.com/optimism",
                    "https://mainnet.optimism.io"
                ],
                "native_token": "ETH",
                "explorer": "https://optimistic.etherscan.io"
            },
            "bsc": {
                "name": "BSC",
                "chain_id": 56,
                "rpc_urls": [
                    "https://bsc.llamarpc.com",
                    "https://rpc.ankr.com/bsc",
                    "https://bsc-dataseed.binance.org"
                ],
                "native_token": "BNB",
                "explorer": "https://bscscan.com"
            },
            "polygon": {
                "name": "Polygon",
                "chain_id": 137,
                "rpc_urls": [
                    "https://polygon.llamarpc.com",
                    "https://rpc.ankr.com/polygon",
                    "https://polygon-rpc.com"
                ],
                "native_token": "POL",
                "explorer": "https://polygonscan.com",
                "etherscan_chain_id": 137  # Для Etherscan API (Polygonscan)
            },
            "arbitrum": {
                "name": "Arbitrum",
                "chain_id": 42161,
                "rpc_urls": [
                    "https://arbitrum.llamarpc.com",
                    "https://rpc.ankr.com/arbitrum",
                    "https://arb1.arbitrum.io/rpc"
                ],
                "native_token": "ETH",
                "explorer": "https://arbiscan.io"
            },
            "avalanche": {
                "name": "Avalanche",
                "chain_id": 43114,
                "rpc_urls": [
                    "https://avalanche.llamarpc.com",
                    "https://rpc.ankr.com/avalanche",
                    "https://api.avax.network/ext/bc/C/rpc"
                ],
                "native_token": "AVAX",
                "explorer": "https://snowtrace.io"
            },
            "base": {
                "name": "Base",
                "chain_id": 8453,
                "rpc_urls": [
                    "https://base.llamarpc.com",
                    "https://rpc.ankr.com/base",
                    "https://mainnet.base.org"
                ],
                "native_token": "ETH",
                "explorer": "https://basescan.org"
            },
            "linea": {
                "name": "Linea",
                "chain_id": 59144,
                "rpc_urls": [
                    "https://linea.llamarpc.com",
                    "https://rpc.ankr.com/linea",
                    "https://rpc.linea.build"
                ],
                "native_token": "ETH",
                "explorer": "https://lineascan.build"
            },
            "scroll": {
                "name": "Scroll",
                "chain_id": 534352,
                "rpc_urls": [
                    "https://scroll.llamarpc.com",
                    "https://rpc.ankr.com/scroll",
                    "https://rpc.scroll.io"
                ],
                "native_token": "ETH",
                "explorer": "https://scrollscan.com"
            },
            "mantle": {
                "name": "Mantle",
                "chain_id": 5000,
                "rpc_urls": [
                    "https://mantle.llamarpc.com",
                    "https://rpc.ankr.com/mantle",
                    "https://rpc.mantle.xyz"
                ],
                "native_token": "MNT",
                "explorer": "https://mantlescan.info"
            },
            "blast": {
                "name": "Blast",
                "chain_id": 81457,
                "rpc_urls": [
                    "https://blast.llamarpc.com",
                    "https://rpc.ankr.com/blast",
                    "https://rpc.blast.io"
                ],
                "native_token": "ETH",
                "explorer": "https://blastscan.io"
            },
            "mode": {
                "name": "Mode",
                "chain_id": 34443,
                "rpc_urls": [
                    "https://mode.llamarpc.com",
                    "https://rpc.ankr.com/mode",
                    "https://mainnet.mode.network"
                ],
                "native_token": "ETH",
                "explorer": "https://modescan.io"
            }
        }
        
        # Текущая сеть по умолчанию
        self.current_network = "polygon"
        
        # Получаем multichain RPC из настроек
        self.multichain_rpc = settings_manager.get("rpc_url_multichain", None)
        
        # Получаем Etherscan API ключ
        self.etherscan_api_key = settings_manager.get_api_key("Etherscan")
        
        # Старые RPC endpoints для совместимости (с приоритетом на multichain)
        self.rpc_endpoints = {
            "ethereum": self.multichain_rpc or "https://mainnet.infura.io/v3/YOUR_INFURA_ID",
            "polygon": self.multichain_rpc or "https://polygon-rpc.com",
        }
    
    def _get_token_contracts_for_network(self, network: str) -> dict:
        """Возвращает контракты токенов для конкретной сети (fallback для динамических токенов)"""
        token_contracts = {
            "polygon": {
                "USDT": "0xc2132D05D31c914a87C6611C10748AEb04B58e8F",
                "USDC": "0x2791Bca1f2de4661ED88A30C99A7a9449Aa84174",
                "DAI": "0x8f3Cf7ad23Cd3CaDbD9735AFf958023239c6A063",
                "WETH": "0x7ceB23fD6bC0adD59E62ac25578270cFf1b9f619",
                "WBTC": "0x1BFD67037B42Cf73acF2047067bd4F2C47D9BfD6",
                "LINK": "0x53E0bca35eC356BD5ddDFebbD1Fc0fD03FaBad39",
            },
            "ethereum": {
                "USDT": "0xdAC17F958D2ee523a2206206994597C13D831ec7",
                "USDC": "0xA0b86991c6218b36c1d19D4a2e9eb0cE3606eB48",
                "DAI": "0x6B175474E89094C44Da98b954EedeAC495271d0F",
                "WBTC": "0x2260FAC5E5542a773Aa44fBCfeDf7C193bc2C599",
                "LINK": "0x514910771AF9Ca656af840dff83E8264EcF986CA",
            },
            "bsc": {
                "USDT": "0x55d398326f99059fF775485246999027B3197955",
                "USDC": "0x8AC76a51cc950d9822D68b83fE1Ad97B32Cd580d",
                "DAI": "0x1AF3F329e8BE154074D8769D1FFa4eE058B1DBc3",
                "WBTC": "0x7130d2A12B9BCbFAe4f2634d864A1Ee1Ce3Ead9c",
                "LINK": "0xF8A0BF9cF54Bb92F17374d9e9A321E6a111a51bD",
            },
            "arbitrum": {
                "USDT": "0xFd086bC7CD5C481DCC9C85ebE478A1C0b69FCbb9",
                "USDC": "0xFF970A61A04b1cA14834A43f5dE4533eBDDB5CC8",
                "DAI": "0xDA10009cBd5D07dd0CeCc66161FC93D7c9000da1",
                "WBTC": "0x2f2a2543B76A4166549F7aaB2e75Bef0aefC5B0f",
                "LINK": "0xf97f4df75117a78c1A5a0DBb814Af92458539FB4",
            },
            "optimism": {
                "USDT": "0x94b008aA00579c1307B0EF2c499aD98a8ce58e58",
                "USDC": "0x7F5c764cBc14f9669B88837ca1490cCa17c31607",
                "DAI": "0xDA10009cBd5D07dd0CeCc66161FC93D7c9000da1",
                "WBTC": "0x68f180fcCe6836688e9084f035309E29Bf0A2095",
                "LINK": "0x350a791Bfc2C21F9Ed5d10980Dad2e2638ffa7f6",
            },
            "avalanche": {
                "USDT": "0x9702230A8Ea53601f5cD2dc00fDBc13d4dF4A8c7",
                "USDC": "0xB97EF9Ef8734C71904D8002F8b6Bc66Dd9c48a6E",
                "DAI": "0xd586E7F844cEa2F87f50152665BCbc2C279D8d70",
                "WBTC": "0x50b7545627a5162F82A992c33b87aDc75187B218",
                "LINK": "0x5947BB275c521040051D82396192181b413227A3",
            },
            "base": {
                "USDC": "0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913",
                "WBTC": "0x4200000000000000000000000000000000000006",
                "DAI": "0x50c5725949A6F0c72E6C4a641F24049A917DB0Cb",
            },
            "linea": {
                "USDC": "0x176211869cA2b568f2A7D4EE941E073a821EE1ff",
                "WBTC": "0x3aAB2285ddcDdaD8edf4385dAc71c08c33cb62B3",
            },
            "scroll": {
                "USDC": "0x06eFdBFf2a14a7c8E15944D1F4A48F9F95F663A4",
                "WBTC": "0x3C1BCa5A656e69edCD0D4E36Bbb3daC244eE6B9D",
            }
        }
        return token_contracts.get(network, {})
    
    def set_network(self, network: str):
        """Устанавливает текущую сеть"""
        if network in self.networks:
            self.current_network = network
            print(f"🌐 Переключено на сеть: {self.networks[network]['name']}")
            return True
        else:
            print(f"❌ Неизвестная сеть: {network}")
            return False
    
    def get_nfts(self, address: str, network: str) -> dict:
        """
        Получает NFT для адреса через Etherscan API
        Возвращает словарь с информацией о NFT
        """
        import requests
        from config.settings import settings_manager
        
        try:
            network_lower = network.lower()
            
            # Маппинг API эндпоинтов
            api_map = {
                'polygon': {
                    'url': 'https://api.etherscan.io/v2/api',
                    'chainid': 137
                },
                'ethereum': {
                    'url': 'https://api.etherscan.io/v2/api',
                    'chainid': 1
                },
                'bsc': {
                    'url': 'https://api.bscscan.com/api',
                    'chainid': 56
                }
            }
            
            if network_lower not in api_map:
                print(f"⚠️ Сеть {network} не поддерживается для NFT")
                return {"nfts": []}
            
            api_info = api_map[network_lower]
            api_key = settings_manager.get_api_key('etherscan')
            if not api_key:
                api_key = self.etherscan_api_key # Используем ключ из настроек
            
            if not api_key or api_key in ['YOUR_ETHERSCAN_API_KEY']:
                print(f"⚠️ API ключ не настроен")
                return {"nfts": []}
            
            print(f"📡 Запрашиваю NFT для адреса через Etherscan...")
            
            # ERC721 (обычные NFT)
            params = {
                'module': 'account',
                'action': 'tokennfttx',  # получить NFT транзакции
                'address': address,
                'page': 1,
                'offset': 100,
                'sort': 'desc',
                'apikey': api_key,
                'chainid': api_info['chainid']
            }
            
            response = requests.get(api_info['url'], params=params, timeout=10)
            data = response.json()
            
            nfts = {}
            
            if data.get('status') == '1':
                result = data.get('result', [])
                print(f"✅ Etherscan вернул {len(result)} NFT транзакций")
                
                # Извлекаем уникальные NFT контракты и их ID
                seen_nfts = set()
                for tx in result:
                    try:
                        # Для токенов, полученных пользователем (to == address)
                        if tx.get('to', '').lower() == address.lower():
                            contract = tx.get('contractAddress', '').lower()
                            token_id = tx.get('tokenID', '')
                            token_name = tx.get('tokenName', 'Unknown')
                            token_symbol = tx.get('tokenSymbol', 'NFT')
                            
                            nft_key = f"{contract}_{token_id}"
                            
                            if nft_key not in seen_nfts:
                                seen_nfts.add(nft_key)
                                if contract not in nfts:
                                    nfts[contract] = {
                                        'symbol': token_symbol,
                                        'name': token_name,
                                        'tokens': []
                                    }
                                
                                nfts[contract]['tokens'].append({
                                    'token_id': token_id,
                                    'contract': contract,
                                    'name': token_name
                                })
                                
                                print(f"  🎨 {token_symbol} (ID: {token_id[:10]}...)")
                    except Exception as e:
                        print(f"  ⚠️ Ошибка парсинга NFT: {e}")
                
                return {"nfts": nfts, "count": len(nfts)}
            else:
                print(f"⚠️ API вернул: {data.get('message')}")
                return {"nfts": []}
                
        except Exception as e:
            print(f"❌ Ошибка получения NFT: {e}")
            return {"nfts": []}
    
    def get_current_network(self) -> dict:
        """Возвращает информацию о текущей сети"""
        return self.networks[self.current_network]
    
    def get_available_networks(self) -> dict:
        """Возвращает список доступных сетей"""
        return {key: info["name"] for key, info in self.networks.items()}
    
    def extract_tokens_from_transactions(self, transactions: list) -> dict:
        """
        Извлекает уникальные токены из истории транзакций с их АДРЕСАМИ контрактов
        Возвращает словарь {символ: адрес_контракта} для всех найденных токенов
        """
        unique_tokens = {}
        
        try:
            print(f"🔍 DEBUG: Начинаю извлечение токенов из {len(transactions)} транзакций")
            
            for idx, tx in enumerate(transactions):
                if idx < 3:  # Выводим первые 3 для отладки
                    print(f"  📋 TX #{idx}: ключи = {list(tx.keys())}")
                
                # Правильные ключи из TransactionScanner: token_symbol, contractAddress
                if 'token_symbol' in tx and tx.get('token_symbol'):
                    symbol = tx.get('token_symbol', '').strip().upper()
                    
                    # ВАЖНО: используем contractAddress из транзакции!
                    contract_addr = tx.get('contractAddress', '').strip()
                    
                    if symbol and contract_addr and contract_addr.lower() != '0x':
                        if symbol not in unique_tokens:
                            unique_tokens[symbol] = contract_addr
                            print(f"  📍 Найден ERC20 токен: {symbol} -> {contract_addr}")
                    elif symbol and not contract_addr:
                        print(f"  ⚠️ Токен {symbol} найден без адреса контракта")
                
                # Для swap'ов: у нас есть from_token и to_token, а также sellToken и buyToken (адреса)
                if tx.get('from_token'):
                    from_symbol = tx.get('from_token', '').strip().upper()
                    sell_token_addr = tx.get('sellToken', '').strip()
                    if from_symbol and sell_token_addr and sell_token_addr.lower() != '0x':
                        if from_symbol not in unique_tokens:
                            unique_tokens[from_symbol] = sell_token_addr
                            print(f"  📍 Найден токен из swap (from): {from_symbol} -> {sell_token_addr}")
                
                if tx.get('to_token'):
                    to_symbol = tx.get('to_token', '').strip().upper()
                    buy_token_addr = tx.get('buyToken', '').strip()
                    if to_symbol and buy_token_addr and buy_token_addr.lower() != '0x':
                        if to_symbol not in unique_tokens:
                            unique_tokens[to_symbol] = buy_token_addr
                            print(f"  📍 Найден токен из swap (to): {to_symbol} -> {buy_token_addr}")
            
            print(f"✅ Всего уникальных токенов найдено в истории: {len(unique_tokens)}")
            print(f"   Найденные токены: {', '.join(sorted(unique_tokens.keys()))}")
            return unique_tokens
            
        except Exception as e:
            import traceback
            print(f"⚠️ Ошибка при извлечении токенов из истории: {e}")
            traceback.print_exc()
            return unique_tokens
    
    def get_balances_for_tokens(self, address: str, tokens_dict: dict, network: str) -> dict:
        """
        Получает балансы для конкретного списка токенов (вместо фиксированного списка)
        
        Args:
            address: адрес кошелька
            tokens_dict: словарь {символ: имя_или_адрес}
            network: название сети (Polygon, Ethereum и т.д.)
        
        Returns:
            Словарь с балансами в формате совместимом с UI
        """
        from web3 import Web3
        from concurrent.futures import ThreadPoolExecutor, as_completed
        
        print(f"💾 Получаю балансы для {len(tokens_dict)} токенов в сети {network}...")
        
        balances = {
            "native_token": {},
            "tokens": {}
        }
        
        try:
            # Проверяем адрес
            if not Web3.is_address(address):
                print(f"❌ Неверный адрес: {address}")
                return balances
            
            address = Web3.to_checksum_address(address)
            
            # Получаем RPC для сети
            network_lower = network.lower()
            rpc_url = None
            rpc_urls_list = []
            
            if network_lower in self.networks:
                network_info = self.networks[network_lower]
                # Используем rpc_urls (массив) вместо rpc (строка)
                rpc_urls_list = network_info.get('rpc_urls', [])
            
            if not rpc_urls_list:
                print(f"⚠️ RPC не найден для сети {network}")
                return balances
            
            # Пытаемся подключиться к одному из доступных RPC'й
            web3 = None
            for rpc_url in rpc_urls_list:
                try:
                    print(f"🔗 Пробую RPC: {rpc_url[:40]}...")
                    web3 = Web3(Web3.HTTPProvider(rpc_url, request_kwargs={'timeout': 3}))
                    if web3.is_connected():
                        print(f"✅ Подключено к {rpc_url[:50]}...")
                        break
                    else:
                        print(f"⚠️ RPC {rpc_url[:40]}... не отвечает")
                        web3 = None
                except Exception as e:
                    print(f"⚠️ Ошибка подключения к {rpc_url[:40]}...: {e}")
                    web3 = None
            
            if not web3:
                print(f"❌ Не удалось подключиться ни к одному RPC для сети {network}")
                return balances
            
            # Получаем нативный баланс
            native_balance = web3.eth.get_balance(address)
            native_decimals = 18  # Стандартные децимали для нативного токена EVM
            native_balance_formatted = native_balance / (10 ** native_decimals)
            
            # Определяем нативный символ
            native_symbol = {
                "ethereum": "ETH",
                "polygon": "POL",
                "bsc": "BNB",
                "arbitrum": "ARB",
                "optimism": "OP",
                "avalanche": "AVAX",
                "base": "ETH"
            }.get(network_lower, "NATIVE")
            
            balances["native_token"] = {
                "symbol": native_symbol,
                "balance": str(native_balance_formatted),
                "decimals": native_decimals
            }
            
            print(f"✅ Нативный баланс ({native_symbol}): {native_balance_formatted:.4f}")
            
            # Получаем адреса контрактов для найденных символов из известного списка
            # Это необходимо, потому что мы извлекли только символы, но не адреса
            token_contracts_for_network = self._get_token_contracts_for_network(network_lower)
            
            # Фильтруем: берем только те токены из извлеченных, которые есть в известном списке
            tokens_to_load = {}
            for symbol, name_or_value in tokens_dict.items():
                # Проверяем есть ли этот символ в известных токенах сети
                if symbol in token_contracts_for_network:
                    contract_address = token_contracts_for_network[symbol]
                    tokens_to_load[symbol] = contract_address
                    print(f"  ✅ Буду загружать баланс {symbol} по адресу {contract_address[:10]}...")
                else:
                    print(f"  ⚠️ Токен {symbol} не найден в списке для сети {network}")
            
            if not tokens_to_load:
                print(f"⚠️ Нет известных токенов для загрузки в этой сети")
                return balances
            
            # Получаем балансы ERC20 токенов параллельно
            def get_token_balance(symbol, contract_address):
                """Получает баланс одного токена"""
                try:
                    if not Web3.is_address(contract_address):
                        return symbol, None
                    
                    contract_address = Web3.to_checksum_address(contract_address)
                    
                    # Создаем контракт
                    contract = web3.eth.contract(address=contract_address, abi=self.erc20_abi)
                    
                    # Получаем баланс и децимали
                    balance = contract.functions.balanceOf(address).call()
                    decimals = 18  # Используем 18 по умолчанию
                    
                    try:
                        decimals = contract.functions.decimals().call()
                    except:
                        pass  # Используем 18 если не можем получить
                    
                    balance_formatted = balance / (10 ** decimals)
                    
                    return symbol, {
                        "balance": str(balance_formatted),
                        "decimals": decimals,
                        "address": contract_address,
                        "raw_balance": str(balance)
                    }
                except Exception as e:
                    print(f"  ⚠️ Ошибка баланса {symbol}: {e}")
                    return symbol, None
            
            # Параллельная загрузка всех токенов
            with ThreadPoolExecutor(max_workers=5) as executor:
                futures = {
                    executor.submit(get_token_balance, symbol, contract_addr): symbol 
                    for symbol, contract_addr in tokens_to_load.items()
                }
                
                for future in as_completed(futures):
                    symbol, balance_data = future.result()
                    if balance_data:
                        balances["tokens"][symbol] = balance_data
                        balance_val = float(balance_data["balance"])
                        if balance_val > 0:
                            print(f"  ✅ {symbol}: {balance_val:.4f}")
            
            print(f"✅ Загружено балансов: {len(balances['tokens'])} токенов")
            return balances
            
        except Exception as e:
            import traceback
            print(f"❌ Ошибка получения балансов: {e}")
            traceback.print_exc()
            return balances

    def _get_erc20_balances_from_etherscan(self, address: str, chain_id: int) -> dict:
        """
        Получает балансы ERC20 токенов через Etherscan API endpoint
        Это более эффективно, чем делать RPC запросы для каждого токена
        Документация: https://docs.etherscan.io/api-reference/endpoint/addresstokenbalance
        """
        import requests
        
        try:
            # Получаем API ключ Etherscan из настроек
            api_keys = settings_manager.get_all_api_keys() if settings_manager else []
            etherscan_key = None
            for key_info in api_keys:
                if key_info.get('provider') == 'Etherscan':
                    etherscan_key = key_info.get('key')
                    break
            
            if not etherscan_key:
                print(f"⚠️ Etherscan API ключ не найден")
                return {}
            
            # URL для Etherscan API
            base_url = "https://api.etherscan.io/v2/api"
            
            params = {
                "chainid": str(chain_id),
                "module": "account",
                "action": "addresstokenbalance",
                "address": address,
                "apikey": etherscan_key
            }
            
            print(f"🔄 Запрашиваю ERC20 балансы через Etherscan API для {address[:10]}...")
            response = requests.get(base_url, params=params, timeout=15)
            response.raise_for_status()
            
            data = response.json()
            
            if data.get('status') == '1' and data.get('result'):
                tokens_data = {}
                for token_info in data.get('result', []):
                    try:
                        symbol = token_info.get('TokenSymbol', '')
                        address_token = token_info.get('TokenAddress', '')
                        balance_raw = int(token_info.get('TokenQuantity', '0'))
                        decimals = int(token_info.get('TokenDivisor', '18'))
                        
                        # Конвертируем из базовых единиц в токены
                        balance = balance_raw / (10 ** decimals)
                        
                        if symbol and balance > 0:  # Только токены с ненулевым балансом
                            tokens_data[symbol] = {
                                "symbol": symbol,
                                "balance": balance,
                                "balance_formatted": f"{balance:.6f}",
                                "contract_address": address_token,
                                "price_usd": None,
                                "value_usd": 0,
                                "icon": f"icons/{symbol.lower()}.png"
                            }
                    except (ValueError, KeyError) as e:
                        print(f"⚠️ Ошибка парсинга токена: {e}")
                        continue
                
                print(f"✅ Получено {len(tokens_data)} ERC20 токенов с ненулевым балансом от Etherscan")
                return tokens_data
            else:
                print(f"⚠️ Etherscan API вернул пустой результат: {data.get('message')}")
                return {}
                
        except Exception as e:
            print(f"❌ Ошибка получения ERC20 балансов от Etherscan: {e}")
            return {}

    def get_token_info_from_etherscan(self, token_symbol: str, network: str) -> dict:
        """
        Получает информацию о токене через Etherscan API
        Ищет токен по символу и возвращает адрес контракта
        
        API: https://docs.etherscan.io/api-reference/endpoint/toptokenholders
        """
        import requests
        
        try:
            # Получаем данные сети
            network_lower = network.lower()
            if network_lower not in self.networks:
                return {}
            
            network_info = self.networks[network_lower]
            base_url = network_info.get('explorer', '').replace('https://', 'https://api.')
            
            # Для разных сетей разные API базы
            api_map = {
                'ethereum': 'https://api.etherscan.io/api',
                'polygon': 'https://api.polygonscan.com/api',
                'bsc': 'https://api.bscscan.com/api',
                'arbitrum': 'https://api.arbiscan.io/api',
                'optimism': 'https://api-optimistic.etherscan.io/api',
                'avalanche': 'https://api.snowtrace.io/api',
                'base': 'https://api.basescan.org/api',
            }
            
            base_url = api_map.get(network_lower)
            if not base_url:
                return {}
            
            # Получаем API ключ
            api_key = settings_manager.get_api_key('etherscan')
            if not api_key:
                api_key = self.etherscan_api_key # Используем ключ из настроек
            
            # Делаем запрос к API поиска токенов
            # К сожалению Etherscan не имеет прямого поиска по символу,
            # но мы можем использовать данные из самих транзакций
            print(f"  🔍 Ищу информацию о токене {token_symbol} на {network}...")
            
            return {}  # Пока возвращаем пусто, так как нет прямого API поиска
            
        except Exception as e:
            print(f"  ⚠️ Ошибка получения информации о токене {token_symbol}: {e}")
            return {}
    
    def get_balances_from_all_tokens(self, address: str, network: str) -> dict:
        """
        Получает ВСЕ токены адреса используя две стратегии:
        1. Извлекает токены из истории транзакций (динамический список)
        2. Добавляет известные популярные токены (fallback)
        """
        from web3 import Web3
        from plugins.transaction_scanner import TransactionScanner
        
        balances = {
            "native_token": {},
            "tokens": {}
        }
        
        try:
            if not Web3.is_address(address):
                print(f"❌ Неверный адрес: {address}")
                return balances
            
            address = Web3.to_checksum_address(address)
            
            # Шаг 1: Получаем ВСЕ токены из истории транзакций
            print("📊 Шаг 1: Загружаю историю для извлечения токенов...")
            tx_scanner = TransactionScanner()
            transactions = tx_scanner.get_transaction_history(address, network)
            print(f"   ✅ Получено {len(transactions)} транзакций")
            
            # Шаг 2: Извлекаем уникальные токены из истории
            print("📊 Шаг 2: Извлекаю уникальные токены из истории...")
            dynamic_tokens = self.extract_tokens_from_transactions(transactions)
            print(f"   ✅ Найдено {len(dynamic_tokens)} уникальных токенов в истории")
            
            # Шаг 3: Добавляем известные токены как fallback
            print("📊 Шаг 3: Добавляю известные популярные токены...")
            token_contracts = self._get_token_contracts_for_network(network.lower())
            
            # Объединяем: сначала динамические, потом дополняем из известных
            all_tokens_to_check = {}
            all_tokens_to_check.update(dynamic_tokens)  # Динамические токены имеют приоритет
            all_tokens_to_check.update(token_contracts)  # Дополняем известными
            
            print(f"   ✅ Всего токенов к проверке: {len(all_tokens_to_check)}")
            
            # Получаем информацию о сети
            network_lower = network.lower()
            if network_lower not in self.networks:
                print(f"⚠️ Сеть {network} не поддерживается")
                return balances
            
            network_info = self.networks[network_lower]
            rpc_urls = network_info.get('rpc_urls', [])
            
            # Подключаемся к RPC
            web3 = None
            for rpc_url in rpc_urls:
                try:
                    web3 = Web3(Web3.HTTPProvider(rpc_url, request_kwargs={'timeout': 3}))
                    if web3.is_connected():
                        print(f"✅ Подключено к RPC для {network}")
                        break
                    web3 = None
                except:
                    pass
            
            if not web3:
                print(f"❌ Не удалось подключиться к RPC для {network}")
                return balances
            
            # Получаем нативный баланс
            native_balance = web3.eth.get_balance(address)
            native_decimals = 18
            native_balance_formatted = native_balance / (10 ** native_decimals)
            
            native_symbol = {
                "ethereum": "ETH",
                "polygon": "POL",
                "bsc": "BNB",
                "arbitrum": "ETH",
                "optimism": "ETH",
                "avalanche": "AVAX",
                "base": "ETH"
            }.get(network_lower, "NATIVE")
            
            balances["native_token"] = {
                "symbol": native_symbol,
                "balance": str(native_balance_formatted),
                "decimals": native_decimals
            }
            
            print(f"✅ Нативный баланс ({native_symbol}): {native_balance_formatted:.4f}")
            
            # Параллельная загрузка всех токенов
            from concurrent.futures import ThreadPoolExecutor, as_completed
            
            def get_token_balance(symbol, contract_address):
                """Получает баланс одного токена"""
                try:
                    if not Web3.is_address(contract_address):
                        return symbol, None
                    
                    contract_address = Web3.to_checksum_address(contract_address)
                    contract = web3.eth.contract(address=contract_address, abi=self.erc20_abi)
                    
                    balance = contract.functions.balanceOf(address).call()
                    decimals = 18
                    
                    try:
                        decimals = contract.functions.decimals().call()
                    except Exception as dec_error:
                        if symbol in ['USDT', 'USDT0']:
                            print(f"    ⚠️ Не смог получить decimals для {symbol}: {dec_error}")
                    
                    balance_formatted = balance / (10 ** decimals)
                    
                    # DEBUG для USDT
                    if symbol in ['USDT', 'USDT0']:
                        print(f"    🔍 {symbol}: raw={balance}, decimals={decimals}, formatted={balance_formatted}")
                    
                    # Возвращаем ВСЕ токены, даже с нулевым балансом
                    return symbol, {
                        "balance": str(balance_formatted),
                        "decimals": decimals,
                        "address": contract_address,
                        "raw_balance": str(balance)
                    }
                    
                except Exception as e:
                    if symbol in ['USDT', 'USDT0']:
                        print(f"    ❌ {symbol}: общая ошибка: {e}")
                    return symbol, None
            
            print(f"📊 Загружаю балансы для {len(all_tokens_to_check)} токенов...")
            
            # ОПТИМИЗАЦИЯ: Увеличиваем воркеры до 20 для быстрой загрузки
            max_workers = min(20, len(all_tokens_to_check))
            with ThreadPoolExecutor(max_workers=max_workers) as executor:
                futures = {
                    executor.submit(get_token_balance, symbol, contract_addr): symbol 
                    for symbol, contract_addr in all_tokens_to_check.items()
                }
                
                for future in as_completed(futures):
                    symbol, balance_data = future.result()
                    if balance_data:
                        balance_val = float(balance_data["balance"])
                        # Показываем ТОЛЬКО токены с ненулевым балансом (как на Polygonscan)
                        if balance_val > 0:
                            balances["tokens"][symbol] = balance_data
                            print(f"  ✅ {symbol}: {balance_val:.8f} (raw={balance_data.get('raw_balance')})")
                        else:
                            status = "⚠️" if symbol in ['USDT', 'USDT0'] else "⚪"
                            print(f"  {status} {symbol}: {balance_val:.8f} (raw={balance_data.get('raw_balance')}) (игнорируем нулевой баланс) - адрес: {balance_data.get('address', 'unknown')}")
            
            print(f"✅ Всего токенов загружено: {len(balances['tokens'])}")
            return balances
            
        except Exception as e:
            import traceback
            print(f"❌ Ошибка: {e}")
            traceback.print_exc()
            return balances
 
    def get_balances(self) -> dict:
        """Получает балансы для текущего адреса и сети."""
        if not self.wallet or not self.wallet.account:
            return {"error": "Кошелек не загружен или адрес не найден."}

        current_network_info = self.networks.get(self.current_network)
        if not current_network_info:
            return {"error": f"Неизвестная сеть: {self.current_network}"}

        rpc_urls = current_network_info["rpc_urls"]
        native_token_symbol = current_network_info["native_token"]
        chain_id = current_network_info["chain_id"]

        # Добавляем multichain RPC в начало списка как приоритетный
        priority_rpcs = []
        if self.multichain_rpc:
            priority_rpcs.append(self.multichain_rpc)
        priority_rpcs.extend(rpc_urls)

        web3 = None
        for rpc_url in priority_rpcs:
            try:
                web3 = Web3(Web3.HTTPProvider(rpc_url, request_kwargs={'timeout': 10}))
                if web3.is_connected():
                    print(f"✅ Подключение к RPC {rpc_url[:50]}... для {self.current_network} успешно")
                    break
                else:
                    print(f"⚠️ RPC {rpc_url[:50]}... для {self.current_network} не отвечает")
            except Exception as e:
                print(f"⚠️ Ошибка подключения к RPC {rpc_url[:50]}... для {self.current_network}: {e}")
                continue

        if not web3 or not web3.is_connected():
            return {"error": f"Не удалось подключиться к сети {current_network_info['name']}"}

        address = self.wallet.account.address
        balances = {
            "network": current_network_info["name"],
            "chain_id": chain_id,
            "native_token": {},
            "tokens": {}
        }

        # Получаем баланс нативного токена (БЕЗ цены - цена загружается асинхронно в UI)
        try:
            balance_wei = web3.eth.get_balance(address)
            balance = web3.from_wei(balance_wei, 'ether')
            # Всегда возвращаем нативный токен, даже если баланс 0
            balances["native_token"] = {
                "symbol": native_token_symbol,
                "balance": float(balance),
                "balance_formatted": f"{float(balance):.6f}",
                "price_usd": None,  # Цена будет получена асинхронно в UI
                "value_usd": 0,  # Будет рассчитано в UI
                "icon": f"icons/{native_token_symbol.lower()}.png"
            }
        except Exception as e:
            print(f"Не удалось получить баланс {native_token_symbol}: {e}")

        # Получаем балансы ERC20 токенов для текущей сети (БЕЗ цен - цены загружаются асинхронно в UI)
        # ОПТИМИЗАЦИЯ: Параллельные запросы через ThreadPoolExecutor
        token_contracts = self._get_token_contracts_for_network(self.current_network)
        
        # ОПТИМИЗАЦИЯ: Фильтруем известные проблемные токены для Polygon
        # COMP и MKR не существуют на Polygon или имеют неверные адреса
        problematic_tokens = ['COMP', 'MKR'] if self.current_network == 'polygon' else []
        token_contracts = {k: v for k, v in token_contracts.items() if k not in problematic_tokens}
        
        # 🆕 УЛУЧШЕНИЕ: Используем Etherscan API для получения ВСЕ ERC20 балансов
        # Это намного эффективнее чем индивидуальные RPC запросы
        print(f"📡 Получаю ERC20 балансы через RPC запросы...")
        
        def get_token_balance(symbol, contract_address, rpc_url, address, current_network_info):
            """Получает баланс одного токена (для параллельного выполнения)"""
            try:
                # ОПТИМИЗАЦИЯ: Еще более уменьшенный таймаут (1 сек вместо 3)
                thread_web3 = Web3(Web3.HTTPProvider(rpc_url, request_kwargs={'timeout': 1}))
                if not thread_web3.is_connected():
                    return None
                
                # Создаем локальный контракт для каждого потока
                contract = thread_web3.eth.contract(
                    address=thread_web3.to_checksum_address(contract_address), 
                    abi=self.erc20_abi
                )
                # ОПТИМИЗАЦИЯ: Таймаут для вызова контракта - очень короткий
                try:
                    balance_wei = contract.functions.balanceOf(address).call()
                    # Пытаемся получить decimals, но не зависаем если ошибка
                    try:
                        decimals = contract.functions.decimals().call()
                    except:
                        decimals = 18  # Fallback на 18 decimals
                except:
                    return None  # Если не можем получить баланс - пропускаем
                
                balance = balance_wei / (10**decimals)
                
                # ОПТИМИЗАЦИЯ: Возвращаем только токены с балансом > 0 (экономия памяти и ускорение UI)
                # Исключение: всегда показываем топ-токены (USDT, USDC, DAI, WETH)
                top_tokens = ['USDT', 'USDC', 'DAI', 'WETH', 'WBTC', 'AAVE', 'LINK', 'UNI']
                if balance > 0 or symbol in top_tokens:
                    return {
                        "symbol": symbol,
                        "balance": float(balance),
                        "balance_formatted": f"{float(balance):.6f}",
                        "price_usd": None,  # Цена будет получена асинхронно в UI через batch запрос
                        "value_usd": 0,  # Будет рассчитано в UI
                        "icon": f"icons/{symbol.lower()}.png"
                    }
                return None  # Пропускаем токены с нулевым балансом (кроме топ-токенов)
            except Exception as e:
                # Не логируем ошибки для токенов, которых может не быть в сети (тихо пропускаем)
                return None
        
        # Параллельная загрузка балансов (максимум 10 потоков одновременно)
        # Используем RPC URL текущего соединения для создания отдельных соединений в потоках
        current_rpc = None
        for rpc_url in priority_rpcs:
            try:
                test_web3 = Web3(Web3.HTTPProvider(rpc_url, request_kwargs={'timeout': 3}))
                if test_web3.is_connected():
                    current_rpc = rpc_url
                    break
            except:
                continue
        
        if not current_rpc:
            current_rpc = priority_rpcs[0] if priority_rpcs else rpc_urls[0]
        
        # ОПТИМИЗАЦИЯ: Используем больше воркеров для параллелизма (20 вместо 10)
        max_workers = min(20, len(token_contracts))
        
        # ОПТИМИЗАЦИЯ: Приоритетные токены показываем в первую очередь
        priority_tokens = ['USDT', 'USDC', 'DAI', 'WETH', 'WBTC']
        sorted_contracts = {
            k: v for k, v in token_contracts.items() 
            if k in priority_tokens
        }
        sorted_contracts.update({
            k: v for k, v in token_contracts.items() 
            if k not in priority_tokens
        })
        
        with ThreadPoolExecutor(max_workers=max_workers) as executor:
            # Создаем задачи для всех токенов (приоритетные сначала)
            future_to_symbol = {
                executor.submit(get_token_balance, symbol, contract_address, current_rpc, address, current_network_info): symbol
                for symbol, contract_address in sorted_contracts.items()
            }
            
            # Собираем результаты по мере их выполнения
            for future in as_completed(future_to_symbol):
                symbol = future_to_symbol[future]
                try:
                    result = future.result()
                    if result:
                        balances["tokens"][result["symbol"]] = {
                            "balance": result["balance"],
                            "balance_formatted": result["balance_formatted"],
                            "price_usd": result["price_usd"],
                            "value_usd": result["value_usd"],
                            "icon": result["icon"]
                        }
                except Exception as e:
                    print(f"⚠️ Ошибка при получении результата для {symbol}: {e}")
        
        # Отладка: выводим все загруженные балансы
        print(f"\n🔍 DEBUG БАЛАНСЫ для {address[:10]}... в {self.current_network}:")
        native = balances.get('native_token', {})
        if native:
            print(f"  Нативный: {native.get('symbol')} = {native.get('balance', 0)}")
        
        tokens = balances.get('tokens', {})
        print(f"  ERC20 токены ({len(tokens)} всего):")
        for symbol, data in sorted(tokens.items()):
            balance = float(data.get('balance', 0))
            status = "✅" if balance > 0 else "❌"
            print(f"    {status} {symbol}: {balance:.4f}")
        
        return balances
    
    def get_balance(self, address: str, token: str, network: str = None) -> float:
        """
        Получает баланс конкретного токена для адреса (совместимость со старым API)
        
        Args:
            address: Адрес кошелька
            token: Символ токена (ETH, POL, USDT и т.д.)
            network: Название сети (polygon, ethereum и т.д.) или None для текущей сети
            
        Returns:
            Баланс токена в виде float
        """
        try:
            # Если сеть не указана, используем текущую
            if network:
                # Конвертируем название сети в формат класса
                network_map = {
                    'ETH': 'ethereum',
                    'Polygon': 'polygon',
                    'BSC': 'bsc',
                    'Ethereum': 'ethereum',
                }
                saved_network = self.current_network
                if network in network_map:
                    self.set_network(network_map[network])
                elif network.lower() in self.networks:
                    self.set_network(network.lower())
            
            # Получаем все балансы
            balances = self.get_balances()
            
            # Восстанавливаем сеть если меняли
            if network:
                self.set_network(saved_network)
            
            # Проверяем наличие ошибки
            if isinstance(balances, dict) and "error" in balances:
                return 0.0
            
            # Проверяем нативный токен
            if 'native_token' in balances and balances['native_token']:
                native = balances['native_token']
                if native.get('symbol', '').upper() == token.upper():
                    return float(native.get('balance', 0))
            
            # Проверяем ERC20 токены
            if 'tokens' in balances and balances['tokens']:
                if token.upper() in balances['tokens']:
                    return float(balances['tokens'][token.upper()].get('balance', 0))
            
            return 0.0
            
        except Exception as e:
            print(f"⚠️ Ошибка получения баланса {token} для {address}: {e}")
            return 0.0


class Plugin:
    """
    Плагин для управления балансами токенов.
    """
    def __init__(self, main_window):
        self.main_window = main_window
        self.dependencies = ["web3", "requests"]
    
    def get_instance(self):
        """Возвращает экземпляр менеджера балансов."""
        if hasattr(self.main_window, 'plugin_manager') and "web3_wallet" in self.main_window.plugin_manager.plugins:
            wallet_plugin = self.main_window.plugin_manager.plugins["web3_wallet"]
            if hasattr(wallet_plugin, 'wallet'):
                return BalanceManager(wallet_plugin.wallet, self.main_window.settings_container)
        
        print("ОШИБКА: Не удалось найти экземпляр кошелька для BalanceManager.")
        return None
