import re
import json
import asyncio
import requests
from typing import Dict, List, Optional, Tuple, Any
from decimal import Decimal
from web3 import Web3
from eth_account import Account

from plugins.dex_integration import DEXIntegration
from plugins.balance_manager import BalanceManager

class AIWalletController:
    """
    Контроллер для AI-управления Web3 кошельком.
    Обрабатывает голосовые и текстовые команды для выполнения операций.
    """
    
    def __init__(self, wallet_instance):
        self.wallet = wallet_instance
        self.dex_integration = DEXIntegration()
        self.balance_manager = BalanceManager(wallet_instance)
        self.current_wallet_name = None
        self.current_address = None
        self.network_rpcs = self._load_network_rpcs()
        self.dex_apis = self._load_dex_apis()
        
        # Добавляем автономные настройки
        self.autonomous_mode = True  # Включаем автономность
        self.risk_thresholds = {
            'max_auto_amount_usd': 1000,  # Максимальная сумма для автоопераций
            'auto_swap_enabled': True,     # Автообмен включен
            'auto_send_enabled': True,     # Автоотправка включена
            'auto_rebalancing_enabled': True  # Автоперебалансировка включена
        }
        
        # Команды для распознавания
        self.command_patterns = {
            'show_balance': [
                r'покажи\s+баланс',
                r'баланс\s+кошелька',
                r'сколько\s+денег',
                r'показать\s+баланс',
                r'покажи\s+баланс\s+кошелька'
            ],
            'swap_tokens': [
                r'обменяй\s+(\d+(?:\.\d+)?)\s+(\w+)\s+на\s+(\w+)',
                r'поменяй\s+(\d+(?:\.\d+)?)\s+(\w+)\s+на\s+(\w+)',
                r'своп\s+(\d+(?:\.\d+)?)\s+(\w+)\s+на\s+(\w+)',
                r'обмен\s+(\d+(?:\.\d+)?)\s+(\w+)\s+на\s+(\w+)',
                r'обменяй\s+(\d+(?:\.\d+)?)\s+долларов\s+на\s+эфир',
                r'обменяй\s+(\d+(?:\.\d+)?)\s+эфира\s+на\s+доллары'
            ],
            'show_tokens': [
                r'покажи\s+токены',
                r'список\s+токенов',
                r'какие\s+токены',
                r'токены\s+в\s+кошельке',
                r'сколько\s+у\s+меня\s+токенов'
            ],
            'send_tokens': [
                r'отправь\s+(\d+(?:\.\d+)?)\s+(\w+)\s+на\s+([a-zA-Z0-9]{42})',
                r'переведи\s+(\d+(?:\.\d+)?)\s+(\w+)\s+на\s+([a-zA-Z0-9]{42})',
                r'отправить\s+(\d+(?:\.\d+)?)\s+(\w+)\s+на\s+([a-zA-Z0-9]{42})',
                r'отправь\s+(\d+(?:\.\d+)?)\s+эфира\s+на\s+адрес'
            ],
            'change_network': [
                r'переключись\s+на\s+(\w+)',
                r'сеть\s+(\w+)',
                r'переключить\s+на\s+(\w+)',
                r'переключись\s+на\s+полигон'
            ],
            'portfolio_rebalancing': [
                r'оптимизируй\s+портфель',
                r'перебалансируй\s+портфель',
                r'сбалансируй\s+портфель',
                r'оптимизация\s+портфеля'
            ]
        }
    
    def _load_network_rpcs(self) -> Dict[str, str]:
        """Загружает RPC URLs для разных сетей"""
        try:
            with open("config/RPC.txt", "r") as f:
                content = f.read()
                rpcs = {}
                
                # Извлекаем RPC URLs
                lines = content.split('\n')
                for line in lines:
                    line = line.strip()
                    if line.startswith('https://'):
                        if 'eth' in line and 'ankr' in line:
                            rpcs['ETH'] = line
                        elif 'polygon' in line:
                            rpcs['Polygon'] = line
                        elif 'base' in line:
                            rpcs['Base'] = line
                        elif 'bsc' in line:
                            rpcs['BSC'] = line
                        elif 'arbitrum' in line:
                            rpcs['Arbitrum'] = line
                        elif 'multichain' in line:
                            rpcs['Multichain'] = line
                
                return rpcs
        except Exception as e:
            print(f"Ошибка загрузки RPC: {e}")
            return {
                'ETH': 'https://mainnet.infura.io/v3/524f929431e44c16a21220aa90e6b253',
                'Polygon': 'https://rpc.ankr.com/polygon/b8d5ff82562be8f50fa93c90fdbfaeeb40bb1d6856bafcd2afb60833774ea0d7'
            }
    
    def _load_dex_apis(self) -> Dict[str, str]:
        """Загружает API ключи для DEX агрегаторов"""
        try:
            with open("config/RPC.txt", "r") as f:
                content = f.read()
                apis = {}
                
                # Ищем API ключи
                for line in content.split('\n'):
                    if '1INCH_API_KEY=' in line:
                        key = line.split('=')[1].strip()
                        if key != 'YOUR_1INCH_API_KEY':
                            apis['1inch'] = key
                
                return apis
        except Exception as e:
            print(f"Ошибка загрузки API ключей: {e}")
            return {}
    
    def set_active_wallet(self, wallet_name: str, address: str):
        """Устанавливает активный кошелек"""
        self.current_wallet_name = wallet_name
        self.current_address = address
        print(f"Активный кошелек: {wallet_name} ({address})")
    
    def parse_command(self, text: str) -> Tuple[str, Dict[str, Any]]:
        """
        Парсит команду и возвращает тип команды и параметры
        """
        text = text.lower().strip()
        
        for command_type, patterns in self.command_patterns.items():
            for pattern in patterns:
                match = re.search(pattern, text)
                if match:
                    params = match.groups() if match.groups() else {}
                    # Преобразуем tuple в dict с индексами
                    params_dict = {f'param_{i}': param for i, param in enumerate(params)}
                    return command_type, params_dict
        
        return 'unknown', {}
    
    def execute_command(self, text: str) -> str:
        """
        Выполняет команду и возвращает результат
        """
        if not self.current_wallet_name or not self.current_address:
            return "Ошибка: Кошелек не разблокирован. Сначала войдите в кошелек."
        
        command_type, params = self.parse_command(text)
        
        try:
            if command_type == 'show_balance':
                return self._show_balance()
            elif command_type == 'swap_tokens':
                return self._swap_tokens(params)
            elif command_type == 'show_tokens':
                return self._show_tokens()
            elif command_type == 'send_tokens':
                return self._send_tokens(params)
            elif command_type == 'change_network':
                return self._change_network(params)
            elif command_type == 'portfolio_rebalancing':
                return self.execute_portfolio_rebalancing()
            else:
                return f"Неизвестная команда: {text}"
        
        except Exception as e:
            print(f"Ошибка выполнения команды: {e}")
            return f"Произошла ошибка: {str(e)}"
    
    def _show_balance(self) -> str:
        """Показывает баланс активного кошелька"""
        try:
            if not self.current_address:
                return "Ошибка: Адрес кошелька не установлен"
                
            # Получаем балансы через новый API
            saved_network = self.balance_manager.current_network
            self.balance_manager.set_network('ethereum')
            balances_data = self.balance_manager.get_balances()
            self.balance_manager.set_network(saved_network)
            
            # Преобразуем в старый формат для совместимости
            balances = {}
            if 'native_token' in balances_data and balances_data['native_token']:
                balances[balances_data['native_token']['symbol']] = balances_data['native_token']['balance']
            if 'tokens' in balances_data:
                for symbol, token_data in balances_data['tokens'].items():
                    balances[symbol] = token_data['balance']
            
            if not balances:
                return f"На кошельке {self.current_wallet_name} нет токенов."

            response = f"Баланс кошелька {self.current_wallet_name}:\n"
            for token, balance in balances.items():
                response += f"{token}: {balance:.6f}\n"
            
            if self.current_address:
                response += f"Адрес: {self.current_address[:10]}...{self.current_address[-6:]}"
            return response
        
        except Exception as e:
            return f"Ошибка получения баланса: {str(e)}"
    
    def _show_tokens(self) -> str:
        """Показывает список токенов"""
        # Используем токены из существующего кошелька
        tokens = [
            "ETH", "USDT", "USDC", "BNB", "POL", "DAI", "LINK", "UNI", "AAVE", "CRV"
        ]
        
        result = f"Токены в кошельке {self.current_wallet_name}:\n"
        for token in tokens[:10]:  # Показываем первые 10
            result += f"• {token}\n"
        
        return result
    
    def _swap_tokens(self, params: Dict[str, Any]) -> str:
        """Выполняет обмен токенов БЕЗ подтверждения в автономном режиме"""
        if len(params) < 3:
            return "Ошибка: Укажите сумму, токен для обмена и токен для получения"
        
        amount_str = params.get('param_0', '0')
        from_token = params.get('param_1', '')
        to_token = params.get('param_2', '')
        
        if not from_token or not to_token:
            return "Ошибка: Укажите токены для обмена"
        
        try:
            amount = float(amount_str)
            if amount <= 0:
                return "Ошибка: Сумма должна быть больше нуля"
            
            # Маппинг русских названий токенов
            token_mapping = {
                'долларов': 'USDT',
                'доллар': 'USDT',
                'эфир': 'ETH',
                'эфира': 'ETH',
                'usdt': 'USDT',
                'eth': 'ETH',
                'matic': 'POL',
                'usdc': 'USDC'
            }
            
            from_token = token_mapping.get(from_token.lower(), from_token.upper())
            to_token = token_mapping.get(to_token.lower(), to_token.upper())
            
            # Проверяем что токены определены
            if not from_token or not to_token:
                return "Ошибка: Не удалось определить токены для обмена"
            
            # Проверяем лимиты автономных операций
            if not self._check_autonomous_limits(amount, from_token):
                return f"Ошибка: Сумма {amount} {from_token} превышает лимит автономных операций"
            
            # Проверяем баланс через BalanceManager
            if not self.current_address:
                return "Ошибка: Адрес кошелька не установлен"
                
            balance = self.balance_manager.get_balance(self.current_address, from_token, 'ETH')
            if amount > balance:
                return f"Ошибка: Недостаточно {from_token}. Доступно: {balance:.6f}"
            
            # В автономном режиме выполняем обмен сразу
            if self.autonomous_mode and self.risk_thresholds['auto_swap_enabled']:
                return self._execute_autonomous_swap(amount, from_token, to_token, 'ETH')
            else:
                # Fallback на старую логику с подтверждением
                return self._prepare_swap_for_confirmation(amount, from_token, to_token, 'ETH')
        
        except ValueError:
            return "Ошибка: Неверный формат суммы"
        except Exception as e:
            return f"Ошибка обмена: {str(e)}"
    
    def _execute_autonomous_swap(self, amount: float, from_token: str, to_token: str, network: str) -> str:
        """Выполняет автономный обмен токенов"""
        try:
            print(f"🚀 Выполняю автономный обмен: {amount} {from_token} → {to_token}")
            
            # Получаем приватный ключ
            private_key = self.wallet.get_private_key_for_session()
            if not private_key:
                return "Ошибка: Не удалось получить приватный ключ. Кошелек заблокирован?"
            
            # Выполняем обмен через DEX интеграцию
            result = self.dex_integration.execute_swap(
                from_token=from_token,
                to_token=to_token,
                amount=amount,
                chain=network,
                private_key=private_key
            )
            
            if result and result.get('success'):
                tx_hash = result.get('tx_hash')
                return f"✅ Автономный обмен выполнен! {amount} {from_token} → {result.get('toTokenAmount', 'N/A')} {to_token}\nХэш: {tx_hash}"
            else:
                error_msg = result.get('error', 'Неизвестная ошибка') if result else 'Неизвестная ошибка'
                return f"❌ Ошибка автономного обмена: {error_msg}"
                
        except Exception as e:
            return f"Ошибка автономного обмена: {str(e)}"
    
    def _prepare_swap_for_confirmation(self, amount: float, from_token: str, to_token: str, network: str) -> str:
        """Подготавливает своп для подтверждения (fallback)"""
        # Сохраняем параметры свопа для подтверждения
        self.pending_swap = {
            'amount': amount,
            'from_token': from_token,
            'to_token': to_token,
            'network': network
        }
        
        # Получаем курс обмена
        quote = self.dex_integration.get_best_quote(from_token, to_token, amount, network)
        if not quote:
            return f"Ошибка: Не удалось получить курс обмена {from_token} на {to_token}"
        
        estimated_amount = quote['toTokenAmount']
        rate = quote['rate']
        
        return f"Обмен {amount} {from_token} на {to_token}:\n" \
               f"Курс: 1 {from_token} = {rate:.6f} {to_token}\n" \
               f"Получите: {estimated_amount:.6f} {to_token}\n" \
               f"Подтвердите операцию голосовой командой 'подтвердить обмен'"
    
    def confirm_swap(self) -> str:
        """Подтверждает и выполняет своп"""
        if not hasattr(self, 'pending_swap') or not self.pending_swap:
            return "Ошибка: Нет ожидающего свопа для подтверждения"
        
        try:
            swap_params = self.pending_swap
            
            private_key = self.wallet.get_private_key_for_session()
            if not private_key:
                return "Ошибка: Не удалось получить приватный ключ. Кошелек заблокирован?"

            result = self.dex_integration.execute_swap(
                from_token=swap_params['from_token'],
                to_token=swap_params['to_token'],
                amount=swap_params['amount'],
                chain=swap_params['network'],
                private_key=private_key
            )

            if result and result.get('success'):
                tx_hash = result.get('tx_hash')
                # Очищаем ожидающий своп
                self.pending_swap = None
                return f"Обмен успешно выполнен! Хэш транзакции: {tx_hash}"
            else:
                error_msg = result.get('error', 'Неизвестная ошибка') if result else 'Неизвестная ошибка'
                return f"Ошибка выполнения свопа: {error_msg}"
        
        except Exception as e:
            return f"Критическая ошибка выполнения свопа: {str(e)}"
    
    def _send_tokens(self, params: Dict[str, Any]) -> str:
        """Отправляет токены на указанный адрес БЕЗ подтверждения в автономном режиме"""
        if len(params) < 3:
            return "Ошибка: Укажите сумму, токен и адрес получателя"
        
        amount = params.get('param_0', '0')
        token = params.get('param_1', '')
        address = params.get('param_2', '')
        
        try:
            amount = float(amount)
            if amount <= 0:
                return "Ошибка: Сумма должна быть больше нуля"
            
            # Проверяем лимиты автономных операций
            if not self._check_autonomous_limits(amount, token):
                return f"Ошибка: Сумма {amount} {token} превышает лимит автономных операций"
            
            # Проверяем формат адреса
            if not Web3.is_address(address):
                return "Ошибка: Неверный формат адреса"

            # Проверяем баланс через BalanceManager
            if not self.current_address:
                return "Ошибка: Адрес кошелька не установлен"
                
            balance = self.balance_manager.get_balance(self.current_address, token.upper(), 'ETH')
            if amount > balance:
                return f"Ошибка: Недостаточно {token.upper()}. Доступно: {balance:.6f}"
            
            # В автономном режиме выполняем отправку сразу
            if self.autonomous_mode and self.risk_thresholds['auto_send_enabled']:
                return self._execute_autonomous_send(amount, token.upper(), address, 'ETH')
            else:
                # Fallback на старую логику с подтверждением
                return self._prepare_send_for_confirmation(amount, token.upper(), address, 'ETH')
        
        except ValueError:
            return "Ошибка: Неверный формат суммы"
        except Exception as e:
            return f"Ошибка отправки: {str(e)}"
    
    def _execute_autonomous_send(self, amount: float, token: str, address: str, network: str) -> str:
        """Выполняет автономную отправку токенов"""
        try:
            print(f"🚀 Выполняю автономную отправку: {amount} {token} на {address[:10]}...")
            
            # Получаем приватный ключ
            private_key = self.wallet.get_private_key_for_session()
            if not private_key:
                return "Ошибка: Не удалось получить приватный ключ. Кошелек заблокирован?"
            
            # Выполняем отправку
            tx_hash = self.wallet.send_transaction(
                private_key,
                address,
                amount,
                token,
                network
            )
            
            if tx_hash:
                return f"✅ Автономная отправка выполнена! {amount} {token} отправлено на {address[:10]}...{address[-6:]}\nХэш: {tx_hash}"
            else:
                return "❌ Ошибка автономной отправки. Проверьте баланс и адрес."
                
        except Exception as e:
            return f"Ошибка автономной отправки: {str(e)}"
    
    def _prepare_send_for_confirmation(self, amount: float, token: str, address: str, network: str) -> str:
        """Подготавливает отправку для подтверждения (fallback)"""
        self.pending_send = {
            'amount': amount,
            'token': token,
            'address': address,
            'network': network
        }
        
        return f"Отправка {amount} {token} на адрес {address[:10]}...{address[-6:]}:\n" \
               f"Подтвердите операцию голосовой командой 'подтвердить отправку'"

    def confirm_send(self) -> str:
        """Подтверждает и выполняет отправку токенов"""
        if not hasattr(self, 'pending_send') or not self.pending_send:
            return "Ошибка: Нет ожидающей отправки для подтверждения"

        try:
            send_params = self.pending_send
            private_key = self.wallet.get_private_key_for_session()
            if not private_key:
                return "Ошибка: Не удалось получить приватный ключ. Кошелек заблокирован?"

            tx_hash = self.wallet.send_transaction(
                private_key,
                send_params['address'],
                send_params['amount'],
                send_params['token'],
                send_params['network']
            )

            if tx_hash:
                self.pending_send = None
                return f"Отправка успешно выполнена! Хэш транзакции: {tx_hash}"
            else:
                return "Ошибка отправки транзакции. Проверьте баланс и адрес."

        except Exception as e:
            return f"Критическая ошибка выполнения отправки: {str(e)}"
    
    def _change_network(self, params: Dict[str, Any]) -> str:
        """Переключает сеть"""
        if not params:
            return "Ошибка: Укажите название сети"
        
        network_param = params.get('param_0', '').lower()
        
        # Маппинг русских названий на английские
        network_mapping = {
            'полигон': 'Polygon',
            'polygon': 'Polygon',
            'эфир': 'ETH',
            'eth': 'ETH',
            'ethereum': 'ETH',
            'bsc': 'BSC',
            'binance': 'BSC',
            'base': 'Base',
            'arbitrum': 'Arbitrum',
            'zksync': 'zkSync'
        }
        
        network = network_mapping.get(network_param, network_param.capitalize())
        available_networks = list(self.network_rpcs.keys())
        
        if network not in available_networks:
            return f"Ошибка: Сеть {network} не поддерживается. Доступные сети: {', '.join(available_networks)}"
        
        return f"Переключение на сеть {network} выполнено"
    
    def _get_exchange_rate(self, from_token: str, to_token: str) -> Optional[float]:
        """Получает курс обмена от DEX агрегаторов"""
        quote = self.dex_integration.get_best_quote(from_token, to_token, 1.0, "ETH")
        if quote:
            return quote['rate']
        return None
    
    def confirm_operation(self, operation_type: str) -> str:
        """Подтверждает операцию"""
        if operation_type == 'swap':
            return self.confirm_swap()
        elif operation_type == 'send':
            return self.confirm_send()
        else:
            return "Операция подтверждена"
    
    def _check_autonomous_limits(self, amount: float, token: str) -> bool:
        """Проверяет лимиты для автономных операций"""
        try:
            # Простая проверка лимитов (можно расширить)
            max_amount = self.risk_thresholds['max_auto_amount_usd']
            
            # Для стейблкоинов проверяем USD эквивалент
            if token.upper() in ['USDT', 'USDC', 'DAI', 'BUSD']:
                return amount <= max_amount
            
            # Для других токенов используем примерную оценку
            # В реальной системе здесь был бы запрос к API цен
            return amount <= max_amount
            
        except Exception as e:
            print(f"Ошибка проверки лимитов: {e}")
            return False  # В случае ошибки запрещаем автооперацию
    
    def execute_portfolio_rebalancing(self) -> str:
        """Выполняет автономную перебалансировку портфеля"""
        try:
            if not self.autonomous_mode or not self.risk_thresholds['auto_rebalancing_enabled']:
                return "Автоматическая перебалансировка отключена"
            
            print("🔄 Выполняю автономную перебалансировку портфеля...")
            
            # Получаем текущие балансы
            if not self.current_address:
                return "Ошибка: Адрес кошелька не установлен"
                
            # Получаем балансы через новый API
            balances_data = self.balance_manager.get_balances()
            
            # Преобразуем в старый формат для совместимости
            balances = {}
            if 'native_token' in balances_data and balances_data['native_token']:
                balances[balances_data['native_token']['symbol']] = balances_data['native_token']['balance']
            if 'tokens' in balances_data:
                for symbol, token_data in balances_data['tokens'].items():
                    balances[symbol] = token_data['balance']
            
            # Простая логика перебалансировки (можно расширить через AI)
            rebalancing_operations = self._generate_rebalancing_plan(balances)
            
            if not rebalancing_operations:
                return "✅ Портфель уже сбалансирован"
            
            executed_operations = []
            
            # Выполняем операции перебалансировки
            for operation in rebalancing_operations:
                if operation['type'] == 'swap':
                    result = self._execute_autonomous_swap(
                        operation['amount'], 
                        operation['from_token'], 
                        operation['to_token'], 
                        operation['network']
                    )
                    executed_operations.append(result)
                
                # Небольшая задержка между операциями
                import time
                time.sleep(2)
            
            return f"✅ Автономная перебалансировка портфеля завершена! Выполнено операций: {len(executed_operations)}"
            
        except Exception as e:
            return f"Ошибка автономной перебалансировки: {str(e)}"
    
    def _generate_rebalancing_plan(self, balances: Dict[str, float]) -> List[Dict[str, Any]]:
        """Генерирует план перебалансировки портфеля"""
        try:
            operations = []
            
            # Простая логика: если есть много одного токена, обмениваем на стейблкоин
            for token, balance in balances.items():
                if balance > 0:
                    # Если токен составляет больше 40% портфеля, обмениваем часть
                    total_value = sum(balances.values())
                    if total_value > 0 and balance / total_value > 0.4:
                        # Обмениваем 20% избытка на USDT
                        excess_amount = balance * 0.2
                        if excess_amount > 0.001:  # Минимальная сумма
                            operations.append({
                                'type': 'swap',
                                'from_token': token,
                                'to_token': 'USDT',
                                'amount': excess_amount,
                                'network': 'ETH'
                            })
            
            return operations
            
        except Exception as e:
            print(f"Ошибка генерации плана перебалансировки: {e}")
            return []


class Plugin:
    """
    Плагин для AI управления кошельком.
    """
    def __init__(self, main_window):
        self.main_window = main_window
        self.dependencies = ["web3", "requests"]
    
    def get_instance(self):
        """Возвращает экземпляр AI контроллера кошелька."""
        return AIWalletController(None)  # wallet_instance будет установлен позже
