import os
import json
import logging
import subprocess
import psutil
from typing import Dict, List, Any, Optional
from datetime import datetime
import traceback

class Plugin:
    """Плагин для автоматической отладки и диагностики AIagent"""
    
    def __init__(self, main_window=None):
        self.main_window = main_window
        self.name = "ai_debugger"
        self.version = "1.0.0"
        self.description = "Автоматическая отладка и диагностика системы"
        
        # Настройка логирования
        self.logger = logging.getLogger(f"AIagent.{self.name}")
        
        # Пути к важным файлам
        self.config_dir = os.path.expanduser("~/.aiaagent")
        self.log_file = os.path.join(self.config_dir, "aiagent.log")
        self.chat_history_dir = os.path.join(self.config_dir, "chat_history")
        
        print(f"🔧 AI Debugger инициализирован")

    def get_plugin_info(self) -> Dict[str, Any]:
        return {
            "name": self.name,
            "version": self.version,
            "description": self.description,
            "tools": [
                "analyze_logs",
                "diagnose_wallet_issues", 
                "check_system_health",
                "fix_configuration_errors",
                "monitor_performance",
                "auto_fix_common_issues"
            ]
        }

    def analyze_logs(self, lines: int = 100) -> Dict[str, Any]:
        """Анализирует логи приложения на предмет ошибок"""
        try:
            if not os.path.exists(self.log_file):
                return {
                    "status": "warning",
                    "message": "Файл логов не найден",
                    "suggestions": ["Проверьте права доступа к директории логов"]
                }
            
            with open(self.log_file, 'r', encoding='utf-8') as f:
                log_lines = f.readlines()[-lines:]  # Последние N строк
            
            errors = []
            warnings = []
            critical_issues = []
            
            for line in log_lines:
                line_lower = line.lower()
                if "error" in line_lower or "exception" in line_lower:
                    errors.append(line.strip())
                elif "warning" in line_lower or "warn" in line_lower:
                    warnings.append(line.strip())
                elif any(keyword in line_lower for keyword in ["critical", "fatal", "crash"]):
                    critical_issues.append(line.strip())
            
            # Анализ паттернов ошибок
            error_patterns = {}
            for error in errors:
                if "rpc для сети" in error.lower():
                    error_patterns["rpc_issues"] = error_patterns.get("rpc_issues", 0) + 1
                elif "timeout" in error.lower():
                    error_patterns["timeout_issues"] = error_patterns.get("timeout_issues", 0) + 1
                elif "connection" in error.lower():
                    error_patterns["connection_issues"] = error_patterns.get("connection_issues", 0) + 1
            
            return {
                "status": "success",
                "analysis": {
                    "total_lines_analyzed": len(log_lines),
                    "errors_found": len(errors),
                    "warnings_found": len(warnings),
                    "critical_issues": len(critical_issues),
                    "error_patterns": error_patterns,
                    "recent_errors": errors[-5:] if errors else [],
                    "recent_warnings": warnings[-3:] if warnings else []
                },
                "recommendations": self._generate_log_recommendations(error_patterns)
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка анализа логов: {str(e)}",
                "traceback": traceback.format_exc()
            }

    def diagnose_wallet_issues(self) -> Dict[str, Any]:
        """Диагностирует проблемы с кошельком"""
        try:
            issues = []
            fixes = []
            
            # Проверка RPC конфигурации
            rpc_config_file = os.path.join(self.config_dir, "RPC.txt")
            if os.path.exists(rpc_config_file):
                with open(rpc_config_file, 'r', encoding='utf-8') as f:
                    rpc_content = f.read()
                    
                if "POLYGON_RPC_URL" not in rpc_content:
                    issues.append("POLYGON_RPC_URL не настроен")
                    fixes.append("Добавить POLYGON_RPC_URL=https://rpc.ankr.com/polygon в RPC.txt")
                elif "rpc.ankr.com/polygon" not in rpc_content:
                    issues.append("RPC URL для Polygon может быть недоступен")
                    fixes.append("Обновить POLYGON_RPC_URL на рабочий эндпоинт")
            else:
                issues.append("Файл RPC.txt не найден")
                fixes.append("Создать файл RPC.txt с настройками RPC")
            
            # Проверка файлов истории кошелька
            wallet_data_dir = os.path.join(self.config_dir, "wallet_data")
            if not os.path.exists(wallet_data_dir):
                issues.append("Директория wallet_data не найдена")
                fixes.append("Создать директорию для данных кошелька")
            
            # Проверка балансов (если есть доступ к плагину)
            balance_issues = self._check_balance_issues()
            if balance_issues:
                issues.extend(balance_issues["issues"])
                fixes.extend(balance_issues["fixes"])
            
            return {
                "status": "success",
                "issues_found": len(issues),
                "issues": issues,
                "suggested_fixes": fixes,
                "severity": "high" if len(issues) > 3 else "medium" if len(issues) > 0 else "low"
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка диагностики кошелька: {str(e)}",
                "traceback": traceback.format_exc()
            }

    def check_system_health(self) -> Dict[str, Any]:
        """Проверяет общее состояние системы"""
        try:
            health_status = {
                "cpu_usage": psutil.cpu_percent(interval=1),
                "memory_usage": psutil.virtual_memory().percent,
                "disk_usage": psutil.disk_usage('/').percent,
                "timestamp": datetime.now().isoformat()
            }
            
            # Проверка процессов AIagent
            aiagent_processes = []
            for proc in psutil.process_iter(['pid', 'name', 'cpu_percent', 'memory_percent']):
                try:
                    if 'python' in proc.info['name'].lower() and 'main.py' in ' '.join(proc.cmdline()):
                        aiagent_processes.append({
                            "pid": proc.info['pid'],
                            "cpu_percent": proc.info['cpu_percent'],
                            "memory_percent": proc.info['memory_percent']
                        })
                except (psutil.NoSuchProcess, psutil.AccessDenied):
                    pass
            
            # Проверка доступности сети
            network_status = self._check_network_connectivity()
            
            # Анализ производительности
            performance_issues = []
            if health_status["cpu_usage"] > 80:
                performance_issues.append("Высокая загрузка CPU")
            if health_status["memory_usage"] > 85:
                performance_issues.append("Высокое потребление памяти")
            if health_status["disk_usage"] > 90:
                performance_issues.append("Мало свободного места на диске")
            
            return {
                "status": "success",
                "system_health": health_status,
                "aiagent_processes": aiagent_processes,
                "network_status": network_status,
                "performance_issues": performance_issues,
                "overall_health": "good" if not performance_issues else "warning" if len(performance_issues) < 3 else "critical"
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка проверки состояния системы: {str(e)}",
                "traceback": traceback.format_exc()
            }

    def fix_configuration_errors(self, issue_type: str = "auto") -> Dict[str, Any]:
        """Автоматически исправляет конфигурационные ошибки"""
        try:
            fixes_applied = []
            errors_fixed = []
            
            if issue_type == "auto":
                # Автоматическое определение и исправление проблем
                
                # Исправление настроек озвучивания
                try:
                    from config.settings import settings_manager
                    voice_enabled = settings_manager.get("voice_enabled", False)
                    speak_responses = settings_manager.get("speak_responses", False)
                    
                    if not voice_enabled:
                        settings_manager.set("voice_enabled", True)
                        fixes_applied.append("Включено озвучивание (voice_enabled=True)")
                        errors_fixed.append("voice_enabled=False")
                    
                    if not speak_responses:
                        settings_manager.set("speak_responses", True)
                        fixes_applied.append("Включено озвучивание ответов (speak_responses=True)")
                        errors_fixed.append("speak_responses=False")
                except Exception as voice_error:
                    print(f"⚠️ Ошибка исправления настроек озвучивания: {voice_error}")
                
                # Исправление RPC конфигурации
                rpc_fix = self._fix_rpc_configuration()
                if rpc_fix["fixed"]:
                    fixes_applied.append(rpc_fix["description"])
                    errors_fixed.append(rpc_fix["error"])
                
                # Исправление настроек плагинов
                plugin_fix = self._fix_plugin_configuration()
                if plugin_fix["fixed"]:
                    fixes_applied.append(plugin_fix["description"])
                    errors_fixed.append(plugin_fix["error"])
                
                # Исправление файлов истории чата
                chat_fix = self._fix_chat_history()
                if chat_fix["fixed"]:
                    fixes_applied.append(chat_fix["description"])
                    errors_fixed.append(chat_fix["error"])
            
            return {
                "status": "success",
                "fixes_applied": len(fixes_applied),
                "fixes": fixes_applied,
                "errors_fixed": errors_fixed,
                "timestamp": datetime.now().isoformat()
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка исправления конфигурации: {str(e)}",
                "traceback": traceback.format_exc()
            }

    def monitor_performance(self, duration: int = 60) -> Dict[str, Any]:
        """Мониторит производительность в реальном времени"""
        try:
            import time
            
            start_time = time.time()
            metrics = []
            
            while time.time() - start_time < duration:
                metric = {
                    "timestamp": datetime.now().isoformat(),
                    "cpu": psutil.cpu_percent(),
                    "memory": psutil.virtual_memory().percent,
                    "disk": psutil.disk_usage('/').percent
                }
                metrics.append(metric)
                time.sleep(5)  # Обновление каждые 5 секунд
            
            # Анализ трендов
            avg_cpu = sum(m["cpu"] for m in metrics) / len(metrics)
            avg_memory = sum(m["memory"] for m in metrics) / len(metrics)
            max_cpu = max(m["cpu"] for m in metrics)
            max_memory = max(m["memory"] for m in metrics)
            
            performance_analysis = {
                "average_cpu": avg_cpu,
                "average_memory": avg_memory,
                "peak_cpu": max_cpu,
                "peak_memory": max_memory,
                "duration_minutes": duration / 60,
                "samples_collected": len(metrics)
            }
            
            return {
                "status": "success",
                "performance_analysis": performance_analysis,
                "metrics": metrics,
                "recommendations": self._generate_performance_recommendations(performance_analysis)
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка мониторинга производительности: {str(e)}",
                "traceback": traceback.format_exc()
            }

    def auto_fix_common_issues(self) -> Dict[str, Any]:
        """Автоматически исправляет наиболее распространенные проблемы"""
        try:
            fixes_applied = []
            
            # 1. Проверка и исправление RPC конфигурации
            rpc_fix = self._ensure_rpc_configuration()
            if rpc_fix:
                fixes_applied.append("RPC конфигурация проверена и исправлена")
            
            # 2. Очистка старых файлов логов
            log_cleanup = self._cleanup_old_logs()
            if log_cleanup["cleaned"]:
                fixes_applied.append(f"Очищено {log_cleanup['files_removed']} старых файлов логов")
            
            # 3. Проверка и создание необходимых директорий
            dir_fix = self._ensure_directories()
            if dir_fix:
                fixes_applied.append("Созданы необходимые директории")
            
            # 4. Оптимизация конфигурации плагинов
            plugin_optimization = self._optimize_plugin_config()
            if plugin_optimization["optimized"]:
                fixes_applied.append("Конфигурация плагинов оптимизирована")
            
            return {
                "status": "success",
                "auto_fixes_applied": len(fixes_applied),
                "fixes": fixes_applied,
                "timestamp": datetime.now().isoformat(),
                "message": f"Применено {len(fixes_applied)} автоматических исправлений"
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка автоматического исправления: {str(e)}",
                "traceback": traceback.format_exc()
            }

    # Вспомогательные методы
    
    def _generate_log_recommendations(self, error_patterns: Dict[str, int]) -> List[str]:
        """Генерирует рекомендации на основе анализа логов"""
        recommendations = []
        
        if error_patterns.get("rpc_issues", 0) > 0:
            recommendations.append("🔧 Обновить RPC URL для Polygon в конфигурации")
            recommendations.append("🔧 Проверить доступность RPC эндпоинтов")
        
        if error_patterns.get("timeout_issues", 0) > 0:
            recommendations.append("⏱️ Увеличить таймауты в api_provider_manager.py")
            recommendations.append("🌐 Проверить стабильность интернет-соединения")
        
        if error_patterns.get("connection_issues", 0) > 0:
            recommendations.append("🔗 Проверить API ключи и их валидность")
            recommendations.append("🌐 Проверить файрвол и сетевые настройки")
        
        if not recommendations:
            recommendations.append("✅ Критических ошибок в логах не обнаружено")
        
        return recommendations

    def _check_balance_issues(self) -> Dict[str, Any]:
        """Проверяет проблемы с балансами"""
        issues = []
        fixes = []
        
        # Проверка файла конфигурации балансов
        balance_config = os.path.join(self.config_dir, "balance_config.json")
        if not os.path.exists(balance_config):
            issues.append("Конфигурация балансов не найдена")
            fixes.append("Создать файл balance_config.json с настройками RPC")
        
        return {"issues": issues, "fixes": fixes}

    def _check_network_connectivity(self) -> Dict[str, Any]:
        """Проверяет сетевое подключение"""
        try:
            import requests
            
            test_urls = [
                "https://api.deepseek.com",
                "https://api.openai.com",
                "https://rpc.ankr.com/polygon"
            ]
            
            connectivity = {}
            for url in test_urls:
                try:
                    response = requests.get(url, timeout=5)
                    connectivity[url] = {
                        "status": "accessible" if response.status_code == 200 else "unreachable",
                        "response_time": response.elapsed.total_seconds()
                    }
                except:
                    connectivity[url] = {"status": "unreachable"}
            
            return connectivity
            
        except Exception as e:
            return {"error": str(e)}

    def _fix_rpc_configuration(self) -> Dict[str, Any]:
        """Исправляет конфигурацию RPC"""
        try:
            rpc_file = os.path.join(self.config_dir, "RPC.txt")
            if not os.path.exists(rpc_file):
                # Создаем файл с базовыми RPC
                os.makedirs(self.config_dir, exist_ok=True)
                with open(rpc_file, 'w', encoding='utf-8') as f:
                    f.write("POLYGON_RPC_URL=https://rpc.ankr.com/polygon\n")
                    f.write("ETH_RPC_URL=https://rpc.ankr.com/eth\n")
                    f.write("BSC_RPC_URL=https://rpc.ankr.com/bsc\n")
                
                return {
                    "fixed": True,
                    "description": "Создан файл RPC.txt с базовыми настройками",
                    "error": "RPC конфигурация отсутствовала"
                }
            
            return {"fixed": False}
            
        except Exception as e:
            return {"fixed": False, "error": str(e)}

    def _fix_plugin_configuration(self) -> Dict[str, Any]:
        """Исправляет конфигурацию плагинов"""
        try:
            plugin_config_file = os.path.join(self.config_dir, "plugin_config.json")
            if not os.path.exists(plugin_config_file):
                # Создаем базовую конфигурацию плагинов
                base_config = {
                    "enabled_plugins": [
                        "balance_manager",
                        "token_price_service", 
                        "web3_wallet",
                        "ai_debugger"
                    ],
                    "disabled_plugins": [
                        "dex_integration",
                        "security_protection",
                        "ai_wallet_controller"
                    ]
                }
                
                os.makedirs(self.config_dir, exist_ok=True)
                with open(plugin_config_file, 'w', encoding='utf-8') as f:
                    json.dump(base_config, f, indent=2, ensure_ascii=False)
                
                return {
                    "fixed": True,
                    "description": "Создана конфигурация плагинов",
                    "error": "Конфигурация плагинов отсутствовала"
                }
            
            return {"fixed": False}
            
        except Exception as e:
            return {"fixed": False, "error": str(e)}

    def _fix_chat_history(self) -> Dict[str, Any]:
        """Исправляет проблемы с историей чата"""
        try:
            if not os.path.exists(self.chat_history_dir):
                os.makedirs(self.chat_history_dir, exist_ok=True)
                return {
                    "fixed": True,
                    "description": "Создана директория для истории чата",
                    "error": "Директория истории чата отсутствовала"
                }
            
            return {"fixed": False}
            
        except Exception as e:
            return {"fixed": False, "error": str(e)}

    def _ensure_rpc_configuration(self) -> bool:
        """Убеждается, что RPC конфигурация существует"""
        try:
            rpc_file = os.path.join(self.config_dir, "RPC.txt")
            if not os.path.exists(rpc_file):
                self._fix_rpc_configuration()
                return True
            return False
        except:
            return False

    def _cleanup_old_logs(self) -> Dict[str, Any]:
        """Очищает старые файлы логов"""
        try:
            files_removed = 0
            if os.path.exists(self.config_dir):
                for filename in os.listdir(self.config_dir):
                    if filename.endswith('.log') and filename != 'aiagent.log':
                        file_path = os.path.join(self.config_dir, filename)
                        if os.path.getsize(file_path) > 50 * 1024 * 1024:  # > 50MB
                            os.remove(file_path)
                            files_removed += 1
            
            return {"cleaned": files_removed > 0, "files_removed": files_removed}
            
        except Exception as e:
            return {"cleaned": False, "error": str(e)}

    def _ensure_directories(self) -> bool:
        """Создает необходимые директории"""
        try:
            directories = [
                self.config_dir,
                self.chat_history_dir,
                os.path.join(self.config_dir, "wallet_data"),
                os.path.join(self.config_dir, "logs")
            ]
            
            created = False
            for directory in directories:
                if not os.path.exists(directory):
                    os.makedirs(directory, exist_ok=True)
                    created = True
            
            return created
            
        except:
            return False

    def _optimize_plugin_config(self) -> Dict[str, Any]:
        """Оптимизирует конфигурацию плагинов"""
        try:
            plugin_config_file = os.path.join(self.config_dir, "plugin_config.json")
            if os.path.exists(plugin_config_file):
                with open(plugin_config_file, 'r', encoding='utf-8') as f:
                    config = json.load(f)
                
                # Добавляем ai_debugger если его нет
                if "ai_debugger" not in config.get("enabled_plugins", []):
                    config.setdefault("enabled_plugins", []).append("ai_debugger")
                    
                    with open(plugin_config_file, 'w', encoding='utf-8') as f:
                        json.dump(config, f, indent=2, ensure_ascii=False)
                    
                    return {"optimized": True}
            
            return {"optimized": False}
            
        except Exception as e:
            return {"optimized": False, "error": str(e)}

    def _generate_performance_recommendations(self, analysis: Dict[str, Any]) -> List[str]:
        """Генерирует рекомендации по производительности"""
        recommendations = []
        
        if analysis["average_cpu"] > 70:
            recommendations.append("🔧 Рассмотрите возможность оптимизации кода или увеличения ресурсов")
        
        if analysis["average_memory"] > 80:
            recommendations.append("🧠 Высокое потребление памяти - проверьте утечки памяти")
        
        if analysis["peak_cpu"] > 95:
            recommendations.append("⚡ Критическая нагрузка на CPU - требуется оптимизация")
        
        if not recommendations:
            recommendations.append("✅ Производительность системы в норме")
        
        return recommendations
