"""
AI Code Fixer - Плагин для автоматического исправления ошибок в коде приложения
"""
import os
import re
import json
import traceback
from typing import Dict, List, Any, Optional
from datetime import datetime


class Plugin:
    """Плагин для автоматического исправления ошибок в коде приложения"""
    
    def __init__(self, main_window=None):
        self.main_window = main_window
        self.name = "ai_code_fixer"
        self.version = "1.0.0"
        self.description = "Автоматическое исправление ошибок в коде приложения"
        
        # Путь к проекту
        self.project_root = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
        
        print(f"🔧 AI Code Fixer инициализирован")
    
    def get_plugin_info(self) -> Dict[str, Any]:
        return {
            "name": self.name,
            "version": self.version,
            "description": self.description,
            "tools": [
                "analyze_code_error",
                "fix_code_error",
                "auto_fix_api_errors",
                "suggest_code_improvements",
                "analyze_code_structure",
                "fix_missing_function_calls"
            ]
        }
    
    def analyze_code_error(self, error_message: str, error_type: str = "auto", file_path: str = "") -> Dict[str, Any]:
        """
        Анализирует ошибку в коде и определяет возможные исправления
        
        Args:
            error_message: Текст ошибки
            error_type: Тип ошибки (api, ui, logic, auto)
            file_path: Путь к файлу с ошибкой (опционально)
        """
        try:
            analysis = {
                "error_type": error_type,
                "error_message": error_message,
                "suggested_fixes": [],
                "file_path": file_path,
                "confidence": 0.0
            }
            
            # Анализ типа ошибки
            error_lower = error_message.lower()
            
            # API ошибки
            if any(keyword in error_lower for keyword in ["403", "401", "unauthorized", "forbidden", "api", "token", "key"]):
                analysis["error_type"] = "api"
                analysis["suggested_fixes"].extend([
                    "Проверить валидность API ключей",
                    "Проверить права доступа API",
                    "Обновить токены доступа",
                    "Исправить обработку ошибок API в коде"
                ])
                analysis["confidence"] = 0.8
            
            # Ошибки обработки данных
            elif any(keyword in error_lower for keyword in ["json", "parse", "decode", "format", "invalid"]):
                analysis["error_type"] = "data_processing"
                analysis["suggested_fixes"].extend([
                    "Добавить проверку формата данных перед парсингом",
                    "Добавить обработку исключений при парсинге",
                    "Исправить формат данных"
                ])
                analysis["confidence"] = 0.7
            
            # Ошибки UI
            elif any(keyword in error_lower for keyword in ["qt", "widget", "ui", "layout", "display"]):
                analysis["error_type"] = "ui"
                analysis["suggested_fixes"].extend([
                    "Проверить инициализацию виджетов",
                    "Исправить размеры и позиционирование",
                    "Добавить проверку существования виджетов"
                ])
                analysis["confidence"] = 0.6
            
            # Ошибки логики
            elif any(keyword in error_lower for keyword in ["none", "attribute", "method", "function", "call"]):
                analysis["error_type"] = "logic"
                analysis["suggested_fixes"].extend([
                    "Добавить проверку на None перед вызовом методов",
                    "Исправить логику условных операторов",
                    "Добавить обработку исключений"
                ])
                analysis["confidence"] = 0.7
            
            # Определение файла с ошибкой
            if not file_path:
                file_path = self._detect_error_file(error_message)
                analysis["file_path"] = file_path
            
            return {
                "status": "success",
                "analysis": analysis,
                "timestamp": datetime.now().isoformat()
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка анализа: {str(e)}",
                "traceback": traceback.format_exc()
            }
    
    def fix_code_error(self, error_message: str, file_path: str = "", fix_type: str = "auto", preview: bool = False) -> Dict[str, Any]:
        """
        Автоматически исправляет ошибку в коде
        
        Args:
            error_message: Текст ошибки
            file_path: Путь к файлу с ошибкой
            fix_type: Тип исправления (auto, api, logic, ui)
            preview: Если True — не перезаписывает файл, а возвращает результат для предпросмотра
        """
        try:
            fixes_applied = []
            
            # Анализируем ошибку
            analysis_result = self.analyze_code_error(error_message, fix_type, file_path)
            if analysis_result["status"] != "success":
                return analysis_result
            
            analysis = analysis_result["analysis"]
            detected_file = analysis["file_path"]
            
            # Если файл не указан, пытаемся найти его
            if not detected_file:
                return {
                    "status": "error",
                    "message": "Не удалось определить файл с ошибкой",
                    "suggestions": analysis["suggested_fixes"]
                }
            
            full_path = os.path.join(self.project_root, detected_file)
            if not os.path.exists(full_path):
                return {
                    "status": "error",
                    "message": f"Файл не найден: {full_path}",
                    "suggestions": analysis["suggested_fixes"]
                }
            
            # Читаем файл
            with open(full_path, 'r', encoding='utf-8') as f:
                original_content = f.read()
            content = original_content
            
            # Применяем исправления в зависимости от типа ошибки
            if analysis["error_type"] == "api":
                fixed_content, fixes = self._fix_api_errors(content, error_message)
                if fixes:
                    fixes_applied.extend(fixes)
                    content = fixed_content
            
            elif analysis["error_type"] == "logic":
                fixed_content, fixes = self._fix_logic_errors(content, error_message)
                if fixes:
                    fixes_applied.extend(fixes)
                    content = fixed_content
            
            elif analysis["error_type"] == "ui":
                fixed_content, fixes = self._fix_ui_errors(content, error_message)
                if fixes:
                    fixes_applied.extend(fixes)
                    content = fixed_content
            
            elif analysis["error_type"] == "data_processing":
                fixed_content, fixes = self._fix_data_processing_errors(content, error_message)
                if fixes:
                    fixes_applied.extend(fixes)
                    content = fixed_content
            
            # Сохраняем исправленный файл (или возвращаем preview)
            if fixes_applied:
                if preview:
                    # Ничего не пишем в файл, просто возвращаем результат
                    return {
                        "status": "preview",
                        "file_path": detected_file,
                        "fixes_applied": fixes_applied,
                        "fixed_content": content,
                        "message": f"PREVIEW: найдено {len(fixes_applied)} исправлений для {detected_file} (файл не изменён)"
                    }

                # Создаем резервную копию ОРИГИНАЛА
                backup_path = f"{full_path}.backup_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
                with open(backup_path, 'w', encoding='utf-8') as f:
                    f.write(original_content)

                # Сохраняем исправленную версию
                with open(full_path, 'w', encoding='utf-8') as f:
                    f.write(content)

                return {
                    "status": "success",
                    "file_path": detected_file,
                    "fixes_applied": fixes_applied,
                    "backup_path": backup_path,
                    "message": f"Исправлено {len(fixes_applied)} ошибок в {detected_file}"
                }
            else:
                return {
                    "status": "warning",
                    "message": "Не удалось автоматически исправить ошибку",
                    "suggestions": analysis["suggested_fixes"]
                }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка исправления кода: {str(e)}",
                "traceback": traceback.format_exc()
            }
    
    def auto_fix_api_errors(self, error_message: str) -> Dict[str, Any]:
        """Автоматически исправляет ошибки API"""
        try:
            # Ищем файлы с обработкой API
            api_files = [
                "plugins/social_media_commands.py",
                "api_provider_manager.py",
                "main.py"
            ]
            
            fixes_applied = []
            
            for file_rel_path in api_files:
                full_path = os.path.join(self.project_root, file_rel_path)
                if not os.path.exists(full_path):
                    continue
                
                with open(full_path, 'r', encoding='utf-8') as f:
                    content = f.read()
                
                # Исправляем типичные ошибки API
                fixed_content, fixes = self._fix_api_errors(content, error_message)
                
                if fixes:
                    # Создаем backup
                    backup_path = f"{full_path}.backup_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
                    with open(backup_path, 'w', encoding='utf-8') as f:
                        f.write(content)
                    
                    # Сохраняем исправления
                    with open(full_path, 'w', encoding='utf-8') as f:
                        f.write(fixed_content)
                    
                    fixes_applied.append({
                        "file": file_rel_path,
                        "fixes": fixes
                    })
            
            return {
                "status": "success",
                "fixes_applied": fixes_applied,
                "message": f"Исправлено {sum(len(f['fixes']) for f in fixes_applied)} ошибок в {len(fixes_applied)} файлах"
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка автоматического исправления API: {str(e)}",
                "traceback": traceback.format_exc()
            }
    
    def suggest_code_improvements(self, file_path: str) -> Dict[str, Any]:
        """Предлагает улучшения кода"""
        try:
            full_path = os.path.join(self.project_root, file_path)
            if not os.path.exists(full_path):
                return {
                    "status": "error",
                    "message": f"Файл не найден: {full_path}"
                }
            
            with open(full_path, 'r', encoding='utf-8') as f:
                content = f.read()
            
            suggestions = []
            
            # Проверка на отсутствие обработки исключений
            if re.search(r'def\s+\w+\([^)]*\):\s*\n\s*[^#\n]*[^except]', content):
                suggestions.append("Добавить обработку исключений в функции")
            
            # Проверка на отсутствие проверок на None
            if re.search(r'\.\w+\([^)]*\)', content) and 'if.*is not None' not in content:
                suggestions.append("Добавить проверки на None перед вызовом методов")
            
            # Проверка на использование print вместо логирования
            if content.count('print(') > 5:
                suggestions.append("Рассмотреть использование logging вместо print")
            
            return {
                "status": "success",
                "file_path": file_path,
                "suggestions": suggestions,
                "timestamp": datetime.now().isoformat()
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка анализа кода: {str(e)}",
                "traceback": traceback.format_exc()
            }
    
    # Вспомогательные методы
    
    def _detect_error_file(self, error_message: str) -> str:
        """Определяет файл с ошибкой по сообщению об ошибке"""
        # Ищем упоминания файлов в traceback
        file_pattern = r'File\s+"([^"]+)"'
        matches = re.findall(file_pattern, error_message)
        if matches:
            file_path = matches[-1]  # Берем последний файл из traceback
            # Преобразуем абсолютный путь в относительный
            if file_path.startswith(self.project_root):
                return os.path.relpath(file_path, self.project_root)
            return file_path
        
        # Ищем упоминания модулей
        module_pattern = r'(\w+\.py)'
        matches = re.findall(module_pattern, error_message)
        if matches:
            # Ищем файл в проекте
            for root, dirs, files in os.walk(self.project_root):
                for file in files:
                    if file == matches[-1]:
                        return os.path.relpath(os.path.join(root, file), self.project_root)
        
        return ""
    
    def _fix_api_errors(self, content: str, error_message: str) -> tuple:
        """Исправляет ошибки API в коде"""
        fixes = []
        fixed_content = content
        
        # Исправление: добавление проверки на 403 ошибки
        if "403" in error_message or "forbidden" in error_message.lower():
            # Ищем места, где обрабатываются ответы API без проверки на 403
            pattern = r'(response\.status_code\s*==\s*200|response\.status_code\s*==\s*201)'
            if re.search(pattern, fixed_content):
                # Добавляем обработку 403 после проверки 200/201
                replacement = r'\1\n                elif response.status_code == 403:\n                    # Обработка ошибки доступа\n                    error_data = response.json() if response.text else {}\n                    error_msg = error_data.get("errors", [{}])[0].get("message", "Доступ запрещен")\n                    raise Exception(f"API вернул 403: {error_msg}")'
                fixed_content = re.sub(pattern, replacement, fixed_content)
                fixes.append("Добавлена обработка ошибки 403")
        
        # Исправление: улучшение обработки JSON ошибок
        if "json" in error_message.lower() or "decode" in error_message.lower():
            # Ищем места, где парсится JSON без try-except
            pattern = r'(\w+)\.json\(\)'
            matches = list(re.finditer(pattern, fixed_content))
            for match in reversed(matches):  # Обрабатываем с конца
                line_start = fixed_content.rfind('\n', 0, match.start())
                context = fixed_content[line_start:match.end()]
                if 'try:' not in context[:200]:  # Если нет try в ближайшем контексте
                    # Добавляем try-except
                    indent = len(context) - len(context.lstrip())
                    try_block = f'\n{" " * indent}try:\n{" " * (indent + 4)}{match.group(0)}\n{" " * indent}except Exception as e:\n{" " * (indent + 4)}print(f"Ошибка парсинга JSON: {{e}}")\n{" " * (indent + 4)}error_data = {{}}'
                    fixed_content = fixed_content[:match.start()] + try_block + fixed_content[match.end():]
                    fixes.append("Добавлена обработка ошибок парсинга JSON")
                    break  # Исправляем только первое вхождение
        
        return fixed_content, fixes
    
    def _fix_logic_errors(self, content: str, error_message: str) -> tuple:
        """Исправляет логические ошибки в коде"""
        fixes = []
        fixed_content = content
        
        # Исправление: добавление проверок на None
        if "none" in error_message.lower() or "attribute" in error_message.lower():
            # Ищем вызовы методов без проверки на None
            pattern = r'(\w+)\.(\w+)\([^)]*\)'
            # Это упрощенная версия - в реальности нужен более сложный парсинг
            # Для демонстрации просто добавляем проверку в критических местах
            if 'if' not in fixed_content or 'is not None' not in fixed_content:
                fixes.append("Рекомендуется добавить проверки на None")
        
        return fixed_content, fixes
    
    def _fix_ui_errors(self, content: str, error_message: str) -> tuple:
        """Исправляет ошибки UI"""
        fixes = []
        fixed_content = content
        
        # Исправление: проверка существования виджетов
        if "widget" in error_message.lower() or "qt" in error_message.lower():
            # Ищем обращения к виджетам без проверки
            pattern = r'self\.(\w+)\s*=\s*(\w+)\(\)'
            if re.search(pattern, fixed_content):
                fixes.append("Рекомендуется добавить проверку инициализации виджетов")
        
        return fixed_content, fixes
    
    def _fix_data_processing_errors(self, content: str, error_message: str) -> tuple:
        """Исправляет ошибки обработки данных"""
        fixes = []
        fixed_content = content
        
        # Добавление проверок формата данных
        if "json" in error_message.lower():
            # Ищем json.loads без try-except
            pattern = r'json\.loads\([^)]+\)'
            if re.search(pattern, fixed_content):
                fixes.append("Рекомендуется добавить обработку исключений при парсинге JSON")
        
        return fixed_content, fixes
    
    def analyze_code_structure(self, function_name: str, file_path: str = "", context: str = "") -> Dict[str, Any]:
        """
        Анализирует структуру кода и находит места, где должна вызываться функция
        
        Args:
            function_name: Имя функции, которую нужно найти (например, "speak_response.emit", "voice_core.speak")
            file_path: Путь к файлу для анализа (если не указан, анализирует main.py)
            context: Контекст использования функции (например, "озвучивание ответов AI")
        """
        try:
            if not file_path:
                file_path = os.path.join(self.project_root, "main.py")
            
            if not os.path.exists(file_path):
                return {
                    "status": "error",
                    "message": f"Файл не найден: {file_path}",
                    "suggestions": []
                }
            
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # Ищем все места, где функция вызывается
            function_patterns = [
                rf'{re.escape(function_name)}\s*\(',
                rf'self\.{re.escape(function_name)}\s*\(',
                rf'\.{re.escape(function_name)}\s*\(',
            ]
            
            call_locations = []
            for pattern in function_patterns:
                for match in re.finditer(pattern, content):
                    line_num = content[:match.start()].count('\n') + 1
                    line_content = content.split('\n')[line_num - 1].strip()
                    call_locations.append({
                        "line": line_num,
                        "content": line_content,
                        "method": self._find_containing_method(content, match.start())
                    })
            
            # Ищем методы, которые должны вызывать функцию, но не вызывают
            missing_calls = []
            
            # Ищем методы, которые обрабатывают ответы AI
            response_methods = [
                r'def\s+on_api_response_received\s*\(',
                r'def\s+on_response_received\s*\(',
            ]
            
            for method_pattern in response_methods:
                for match in re.finditer(method_pattern, content):
                    method_start = match.start()
                    method_end = self._find_method_end(content, method_start)
                    method_content = content[method_start:method_end]
                    method_name = match.group(0).split('(')[0].replace('def', '').strip()
                    
                    # Проверяем, вызывается ли функция в этом методе
                    has_call = any(re.search(pattern, method_content) for pattern in function_patterns)
                    
                    if not has_call:
                        # Проверяем, должен ли этот метод вызывать функцию
                        should_call = self._should_method_call_function(method_content, context, function_name)
                        
                        if should_call:
                            line_num = content[:method_start].count('\n') + 1
                            missing_calls.append({
                                "method": method_name,
                                "line": line_num,
                                "reason": self._get_missing_call_reason(method_content, context, function_name),
                                "suggested_location": self._find_insert_location(method_content)
                            })
            
            return {
                "status": "success",
                "function_name": function_name,
                "file_path": file_path,
                "call_locations": call_locations,
                "missing_calls": missing_calls,
                "analysis": {
                    "total_calls": len(call_locations),
                    "missing_calls_count": len(missing_calls),
                    "recommendation": "Добавить вызов функции в методы обработки ответов" if missing_calls else "Все необходимые вызовы присутствуют"
                }
            }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка анализа структуры кода: {str(e)}",
                "traceback": traceback.format_exc()
            }
    
    def fix_missing_function_calls(self, function_name: str, file_path: str = "", context: str = "") -> Dict[str, Any]:
        """
        Автоматически добавляет пропущенные вызовы функций в код
        
        Args:
            function_name: Имя функции для добавления (например, "speak_response.emit")
            file_path: Путь к файлу (если не указан, используется main.py)
            context: Контекст использования функции
        """
        try:
            # Сначала анализируем структуру
            analysis = self.analyze_code_structure(function_name, file_path, context)
            
            if analysis["status"] != "success":
                return analysis
            
            if not analysis["missing_calls"]:
                return {
                    "status": "success",
                    "message": "Все необходимые вызовы функции уже присутствуют",
                    "fixes_applied": 0
                }
            
            if not file_path:
                file_path = os.path.join(self.project_root, "main.py")
            
            if not os.path.exists(file_path):
                return {
                    "status": "error",
                    "message": f"Файл не найден: {file_path}"
                }
            
            # Читаем файл
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # Создаем резервную копию
            backup_path = f"{file_path}.backup_{datetime.now().strftime('%Y%m%d_%H%M%S')}"
            with open(backup_path, 'w', encoding='utf-8') as f:
                f.write(content)
            
            fixes_applied = []
            
            # Добавляем вызовы функций в нужные места
            for missing_call in analysis["missing_calls"]:
                method_name = missing_call["method"]
                
                # Ищем метод в коде
                method_pattern = rf'def\s+{re.escape(method_name)}\s*\('
                match = re.search(method_pattern, content)
                
                if match:
                    method_start = match.start()
                    method_end = self._find_method_end(content, method_start)
                    method_content = content[method_start:method_end]
                    
                    # Находим место для вставки
                    insert_location = missing_call["suggested_location"]
                    if insert_location:
                        # Определяем отступ
                        lines = method_content.split('\n')
                        insert_line_idx = insert_location.get("line_index", len(lines) - 2)
                        if insert_line_idx < len(lines):
                            indent = self._get_indent(lines[insert_line_idx] if insert_line_idx > 0 else lines[0])
                            
                            # Формируем код для вставки
                            call_code = self._generate_function_call(function_name, context, indent)
                            
                            # Вставляем код
                            method_lines = method_content.split('\n')
                            method_lines.insert(insert_line_idx + 1, call_code)
                            new_method_content = '\n'.join(method_lines)
                            
                            # Заменяем метод в основном коде
                            content = content[:method_start] + new_method_content + content[method_end:]
                            
                            fixes_applied.append({
                                "method": method_name,
                                "line": content[:method_start].count('\n') + 1,
                                "call": call_code.strip()
                            })
            
            # Сохраняем исправленный файл
            if fixes_applied:
                with open(file_path, 'w', encoding='utf-8') as f:
                    f.write(content)
                
                return {
                    "status": "success",
                    "file_path": file_path,
                    "backup_path": backup_path,
                    "fixes_applied": fixes_applied,
                    "message": f"Добавлено {len(fixes_applied)} вызовов функции {function_name}"
                }
            else:
                return {
                    "status": "warning",
                    "message": "Не удалось автоматически добавить вызовы функции",
                    "suggestions": [mc["reason"] for mc in analysis["missing_calls"]]
                }
            
        except Exception as e:
            return {
                "status": "error",
                "message": f"Ошибка добавления вызовов функции: {str(e)}",
                "traceback": traceback.format_exc()
            }
    
    def _find_containing_method(self, content: str, position: int) -> str:
        """Находит метод, содержащий указанную позицию"""
        before = content[:position]
        lines = before.split('\n')
        
        for line in reversed(lines):
            if re.match(r'\s*def\s+\w+', line):
                method_match = re.search(r'def\s+(\w+)', line)
                if method_match:
                    return method_match.group(1)
        return "unknown"
    
    def _find_method_end(self, content: str, method_start: int) -> int:
        """Находит конец метода (по отступам)"""
        lines = content[method_start:].split('\n')
        if not lines:
            return method_start
        
        first_line = lines[0]
        base_indent = len(first_line) - len(first_line.lstrip())
        
        for i, line in enumerate(lines[1:], 1):
            stripped = line.lstrip()
            if stripped and not stripped.startswith('#'):
                current_indent = len(line) - len(line.lstrip())
                if current_indent <= base_indent:
                    return method_start + sum(len(l) + 1 for l in lines[:i])
        
        return len(content)
    
    def _should_method_call_function(self, method_content: str, context: str, function_name: str) -> bool:
        """Определяет, должен ли метод вызывать функцию"""
        method_lower = method_content.lower()
        
        # Если контекст связан с озвучиванием
        if any(keyword in context.lower() for keyword in ["озвучивание", "voice", "speak"]):
            # Проверяем, обрабатывает ли метод ответы
            if any(keyword in method_lower for keyword in ["response", "answer", "message", "chat"]):
                # Проверяем, что метод добавляет сообщение в чат
                if "add_message" in method_lower or "chat_area" in method_lower:
                    return True
        
        return False
    
    def _get_missing_call_reason(self, method_content: str, context: str, function_name: str) -> str:
        """Возвращает причину, почему метод должен вызывать функцию"""
        if "response" in method_content.lower() and "add_message" in method_content.lower():
            return f"Метод обрабатывает ответы AI и добавляет их в чат, но не вызывает {function_name} для озвучивания"
        return f"Метод должен вызывать {function_name} согласно контексту: {context}"
    
    def _find_insert_location(self, method_content: str) -> Dict[str, Any]:
        """Находит место для вставки вызова функции"""
        lines = method_content.split('\n')
        
        # Ищем место после add_message или перед restore_send_button
        for i, line in enumerate(lines):
            if "add_message" in line.lower() or "chat_area" in line.lower():
                # Ищем конец блока (пустая строка или следующий блок)
                for j in range(i + 1, min(i + 5, len(lines))):
                    if not lines[j].strip() or lines[j].strip().startswith('#'):
                        continue
                    if "restore_send_button" in lines[j] or "auto_save" in lines[j]:
                        return {"line_index": j - 1}
                return {"line_index": i + 1}
        
        # Если не нашли, вставляем перед последней строкой
        return {"line_index": len(lines) - 2}
    
    def _get_indent(self, line: str) -> str:
        """Получает отступ строки"""
        return len(line) - len(line.lstrip())
    
    def _generate_function_call(self, function_name: str, context: str, indent: str) -> str:
        """Генерирует код вызова функции"""
        # Определяем, какой тип вызова нужен
        if "speak_response" in function_name:
            # Генерируем вызов для озвучивания ответов
            code = f'{indent}# Озвучиваем ответ, если включено\n'
            code += f'{indent}if filtered_response and settings_manager.get("speak_responses", True):\n'
            code += f'{indent}    # Сигнализируем голограмме, что AIagent говорит\n'
            code += f'{indent}    if hasattr(self, \'ai_speaking_signal\'):\n'
            code += f'{indent}        self.ai_speaking_signal.emit()\n'
            code += f'{indent}    print(f"🔊 Отправляю сигнал озвучки: {{filtered_response[:50]}}...")\n'
            code += f'{indent}    self.speak_response.emit(filtered_response)\n'
            code += f'{indent}else:\n'
            code += f'{indent}    if not filtered_response:\n'
            code += f'{indent}        print("🔇 Ответ пустой, озвучка пропущена")\n'
            code += f'{indent}    else:\n'
            code += f'{indent}        print(f"🔇 Озвучка отключена в настройках (speak_responses={{settings_manager.get(\'speak_responses\', False)}})")\n'
        else:
            # Общий случай
            code = f'{indent}# Вызываем {function_name}\n'
            if "self." not in function_name and "." not in function_name:
                code += f'{indent}self.{function_name}()\n'
            else:
                code += f'{indent}{function_name}\n'
        
        return code

