"""
Менеджер настроек приложения
"""
import json
import os
from pathlib import Path
from typing import Optional

# Импортируем модуль шифрования только при необходимости
api_key_encryption = None

def get_encryption():
    """Получает модуль шифрования по требованию"""
    global api_key_encryption
    if api_key_encryption is None:
        try:
            from .encryption import api_key_encryption as enc
            api_key_encryption = enc
        except ImportError:
            # Если модуль не найден, создаем заглушку
            class DummyEncryption:
                def encrypt_api_key(self, key, provider=None): return key
                def decrypt_api_key(self, key, provider=None): return key
                def is_encrypted(self, key): return False
            api_key_encryption = DummyEncryption()
    return api_key_encryption

class SettingsManager:
    """Менеджер настроек приложения"""
    
    def __init__(self):
        self.config_file = Path("config/settings.json")
        self.settings = self.load_default_settings()
        self.load_settings()
    
    def load_default_settings(self):
        """Загружает настройки по умолчанию"""
        return {
            "language": "ru",
            "voice_enabled": True,
            "voice_engine": "piper",
            "voice_speed": 1.0,
            "theme": "dark",
            "auto_save": True,
            "log_level": "INFO",
            "max_tokens": 8000,
            "temperature": 0.7,
            "model_path": "models/Qwen3-4B.Q4_K_M.gguf",
            "zerox_fee": {
                "recipient": "0xa89E857ae6eD1C145F81d4c9eBF409B855c73820",
                "buy_token_percentage": 0.0005,
                "sell_token_percentage": 0.0,
                "affiliate_address": "0xa89E857ae6eD1C145F81d4c9eBF409B855c73820"
            },
            "social_credentials": {}
        }
    
    def load_settings(self):
        """Загружает настройки из файла"""
        if self.config_file.exists():
            try:
                with open(self.config_file, 'r', encoding='utf-8') as f:
                    loaded_settings = json.load(f)
                    self.settings.update(loaded_settings)
            except Exception as e:
                print(f"Ошибка загрузки настроек: {e}")
        self._ensure_zerox_fee_defaults()
        if "social_credentials" not in self.settings or not isinstance(self.settings.get("social_credentials"), dict):
            self.settings["social_credentials"] = {}
            self.save_settings()
    
    def _ensure_zerox_fee_defaults(self):
        """Гарантирует наличие настроек комиссии 0x"""
        default_fee = {
            "recipient": "0xa89E857ae6eD1C145F81d4c9eBF409B855c73820",
            "buy_token_percentage": 0.0005,
            "sell_token_percentage": 0.0,
            "affiliate_address": "0xa89E857ae6eD1C145F81d4c9eBF409B855c73820"
        }
        fee_cfg = self.settings.get("zerox_fee")
        changed = False
        if not isinstance(fee_cfg, dict):
            self.settings["zerox_fee"] = default_fee.copy()
            changed = True
        else:
            for key, value in default_fee.items():
                if key not in fee_cfg or fee_cfg.get(key) in (None, "", 0, 0.0):
                    fee_cfg[key] = value
                    changed = True
        if changed:
            self.save_settings()
    
    def get_zerox_fee_config(self, wallet_address: Optional[str] = None) -> dict:
        """Возвращает (и при необходимости обновляет) конфигурацию комиссии 0x"""
        self._ensure_zerox_fee_defaults()
        fee_cfg = dict(self.settings.get("zerox_fee", {}))
        updated = False
        default_buy = 0.0005
        
        if wallet_address:
            if fee_cfg.get("recipient") != wallet_address:
                fee_cfg["recipient"] = wallet_address
                updated = True
            if fee_cfg.get("affiliate_address") != wallet_address:
                fee_cfg["affiliate_address"] = wallet_address
                updated = True
        
        if not fee_cfg.get("buy_token_percentage"):
            fee_cfg["buy_token_percentage"] = default_buy
            updated = True
        
        if "sell_token_percentage" not in fee_cfg:
            fee_cfg["sell_token_percentage"] = 0.0
            updated = True
        
        if updated:
            self.settings["zerox_fee"] = fee_cfg
            self.save_settings()
        
        return fee_cfg
    
    def save_settings(self):
        """Сохраняет настройки в файл"""
        try:
            self.config_file.parent.mkdir(parents=True, exist_ok=True)
            with open(self.config_file, 'w', encoding='utf-8') as f:
                json.dump(self.settings, f, indent=2, ensure_ascii=False)
        except Exception as e:
            print(f"Ошибка сохранения настроек: {e}")
    
    def get(self, key, default=None):
        """Получает значение настройки"""
        return self.settings.get(key, default)
    
    def set(self, key, value):
        """Устанавливает значение настройки"""
        self.settings[key] = value
        self.save_settings()
    
    def update(self, new_settings):
        """Обновляет несколько настроек"""
        self.settings.update(new_settings)
        self.save_settings()
    
    def set_social_credentials(self, platform: str, credentials: dict):
        """Сохраняет учетные данные для соцсетей"""
        if not platform or not isinstance(credentials, dict):
            return
        platform_key = platform.lower().strip()
        social_data = self.settings.setdefault("social_credentials", {})
        # Оставляем только ожидаемые поля
        allowed_keys = {
            "api_key",
            "api_secret",
            "bearer_token",
            "access_token",
            "access_token_secret"
        }
        filtered = {
            key: value.strip()
            for key, value in credentials.items()
            if key in allowed_keys and isinstance(value, str) and value.strip()
        }
        social_data[platform_key] = filtered
        self.save_settings()
    
    def get_social_credentials(self, platform: str) -> dict:
        """Возвращает учетные данные соцсетей"""
        if not platform:
            return {}
        platform_key = platform.lower().strip()
        social_data = self.settings.get("social_credentials", {}) or {}
        creds = social_data.get(platform_key, {})
        return dict(creds) if isinstance(creds, dict) else {}
    
    def remove_api_key(self, provider):
        """Удаляет API ключ для указанного провайдера"""
        if "api_keys" in self.settings:
            self.settings["api_keys"] = [
                key for key in self.settings["api_keys"] 
                if key.get("provider") != provider
            ]
            self.save_settings()
    
    def add_api_key(self, provider: str, api_key: str, encrypted: bool = False):
        """Добавляет API ключ (шифрует только Web3, AI оставляет как есть)"""
        if "api_keys" not in self.settings:
            self.settings["api_keys"] = []
        
        # Проверяем, есть ли уже ключ для этого провайдера
        for key_data in self.settings["api_keys"]:
            if key_data.get("provider") == provider:
                key_data["key"] = api_key
                # AI провайдеры не шифруются
                web3_providers = ["Etherscan", "BSCScan", "CoinGecko", "0x"]
                if provider in web3_providers:
                    key_data["encrypted"] = encrypted
                else:
                    key_data["encrypted"] = False
                break
        else:
            # Добавляем новый ключ
            web3_providers = ["Etherscan", "BSCScan", "CoinGecko", "0x"]
            is_web3 = provider in web3_providers
            self.settings["api_keys"].append({
                "provider": provider,
                "key": api_key,
                "encrypted": encrypted if is_web3 else False
            })
        
        self.save_settings()
    
    def get_api_key(self, provider: str) -> str | None:
        """Получает API ключ для указанного провайдера"""
        if "api_keys" not in self.settings:
            return None
        
        for key_data in self.settings["api_keys"]:
            if key_data and key_data.get("provider") == provider:
                return key_data.get("key")
        return None
    
    def cleanup_api_keys(self):
        """Удаляет все API ключи кроме Deepseek и OpenAI"""
        if "api_keys" not in self.settings:
            return
        
        allowed_providers = ["Deepseek", "OpenAI"]
        original_count = len(self.settings["api_keys"])
        
        # Фильтруем, оставляя только разрешенные провайдеры
        self.settings["api_keys"] = [
            key for key in self.settings["api_keys"]
            if key and key.get("provider") in allowed_providers
        ]
        
        removed_count = original_count - len(self.settings["api_keys"])
        
        if removed_count > 0:
            print(f"🧹 Очищены API ключи: удалено {removed_count}, осталось {len(self.settings['api_keys'])}")
            # Сохраняем изменения
            self.save_settings()
        else:
            print(f"✅ API ключи уже очищены: только Deepseek и OpenAI")
        
        # Выводим оставшихся провайдеров
        providers = [key.get("provider") for key in self.settings["api_keys"] if key]
        if providers:
            print(f"📌 Активные API провайдеры: {', '.join(providers)}")
        else:
            print(f"⚠️ Нет активных API ключей Deepseek и OpenAI")
    
    def get_all_api_keys(self) -> list:
        """Получает все API ключи (расшифровывает только Web3)"""
        print(f"🔍 Проверяю наличие api_keys в настройках: {'api_keys' in self.settings}")
        print(f"🔍 Все ключи настроек: {list(self.settings.keys())}")
        if "api_keys" not in self.settings:
            print("❌ api_keys не найден в настройках")
            return []
        
        result_keys = []
        for key_data in self.settings["api_keys"]:
            provider = key_data.get("provider")
            key = key_data.get("key")
            if provider and key:
                web3_providers = ["Etherscan", "BSCScan", "CoinGecko", "0x"]
                if provider in web3_providers and key_data.get("encrypted", False):
                    # Web3 ключи расшифровываем
                    encryption = get_encryption()
                    decrypted_key = encryption.decrypt_api_key(key, provider)
                    result_keys.append({
                        "provider": provider,
                        "key": decrypted_key,
                        "encrypted": True
                    })
                else:
                    # AI API ключи оставляем как есть
                    result_keys.append({
                        "provider": provider,
                        "key": key,
                        "encrypted": False
                    })
        
        return result_keys
    
    def migrate_existing_keys(self):
        """Мигрирует существующие ключи (шифрует только Web3)"""
        if "api_keys" not in self.settings:
            return
        
        migrated = False
        web3_providers = ["Etherscan", "BSCScan", "CoinGecko", "0x"]
        
        for key_data in self.settings["api_keys"]:
            provider = key_data.get("provider")
            if not key_data.get("encrypted", False) and provider in web3_providers:
                # Шифруем только Web3 ключи
                old_key = key_data.get("key")
                if old_key:
                    encryption = get_encryption()
                    encrypted_key = encryption.encrypt_api_key(old_key, provider)
                    key_data["key"] = encrypted_key
                    key_data["encrypted"] = True
                    migrated = True
                    print(f"✅ Web3 API ключ {provider} зашифрован")
        
        if migrated:
            self.save_settings()
            print("✅ Web3 API ключи зашифрованы")
        else:
            print("ℹ️ Миграция не требуется - все Web3 ключи уже зашифрованы")

# Глобальный экземпляр менеджера настроек
settings_manager = SettingsManager()







